import React, {useCallback, useEffect, useState} from 'react';
import {Redirect} from 'react-router-dom';
import {useDispatch} from 'react-redux';

import {EAviaGoal} from 'utilities/metrika/types/goals/avia';

import {confirmationComplete} from 'reducers/avia/booking/orderPayment/actions';

import {reachGoal} from 'utilities/metrika';
import scrollTo from 'utilities/dom/scrollTo';
import {aviaURLs} from 'projects/avia/lib/urls';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import IRedirectTargetLoadedMessage from 'projects/avia/pages/AviaBookingPaymentRedirectTarget/utilities/IRedirectTargetLoadedMessage';

import {BookingPaymentConfirmationScreen} from 'projects/avia/pages/AviaBookingPayment/components/BookingPaymentConfirmationScreen/BookingPaymentConfirmationScreen';
import {BookingPaymentsScreen} from 'projects/avia/pages/AviaBookingPayment/components/BookingPaymentsScreen/BookingPaymentsScreen';
import AviaPaymentLayout from 'projects/avia/pages/AviaBookingPayment/components/AviaPaymentLayout/AviaPaymentLayout';

import {IAviaBookingOrderPageData} from 'server/services/AviaBookingService/order/IAviaBookingOrderPageData';

const REDIRECT_TIMEOUT = 60 * 1000;

interface IBookingPaymentContentProps {
    priceIsVerified: boolean;
    confirmationUrl: string | null;
    paymentUrl: string | null;
    orderId: string;
    order: IAviaBookingOrderPageData | null | undefined;
}

export const BookingPaymentContent: React.FC<IBookingPaymentContentProps> = ({
    priceIsVerified,
    confirmationUrl,
    paymentUrl,
    orderId,
    order,
}) => {
    const {value: confirmationIsDone, setTrue: setConfirmationIsDone} =
        useBoolean(false);
    const [forceRedirectToOrder, setForceRedirectToOrder] = useState(false);

    const dispatch = useDispatch();

    useEffect(() => {
        scrollTo({top: 0});
        reachGoal(EAviaGoal.BOY_PAYMENT_PAGE_OPEN);
    }, []);

    useEffect(() => {
        if (!confirmationIsDone) {
            return undefined;
        }

        const handle = setTimeout(() => {
            setForceRedirectToOrder(true);
        }, REDIRECT_TIMEOUT);

        return (): void => {
            clearTimeout(handle);
        };
    }, [confirmationIsDone]);

    const handleConfirmationIsDone = useCallback(
        (message: IRedirectTargetLoadedMessage): void => {
            setConfirmationIsDone();

            dispatch(
                confirmationComplete({
                    id: orderId,
                    success: message.payload.success,
                }),
            );
        },
        [setConfirmationIsDone, orderId, dispatch],
    );

    if (forceRedirectToOrder) {
        return <Redirect to={aviaURLs.getBookingResultUrl(orderId)} />;
    }

    const spinner = (
        <AviaPaymentLayout
            showFooter={false}
            showOrderInfo={false}
            paymentIsStarted={false}
            order={order}
            loading
            {...prepareQaAttributes('aviaBookingLayout')}
        />
    );

    if (!priceIsVerified) {
        return spinner;
    }

    if (confirmationUrl) {
        return (
            <BookingPaymentConfirmationScreen
                confirmationIsDone={confirmationIsDone}
                confirmationUrl={confirmationUrl}
                onDone={handleConfirmationIsDone}
                order={order}
            />
        );
    }

    if (paymentUrl) {
        return <BookingPaymentsScreen paymentUrl={paymentUrl} order={order} />;
    }

    return spinner;
};
