import React, {useMemo} from 'react';
import moment from 'moment';
import _head from 'lodash/head';
import _last from 'lodash/last';
import _flatMap from 'lodash/flatMap';

import {IWithClassName} from 'types/withClassName';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {HUMAN, ROBOT} from 'utilities/dateUtils/formats';
import {CHAR_NBSP} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/trains-order-payment';

import Price from 'components/Price/Price';
import PaymentOrderInfo from 'components/Layouts/PaymentLayout/components/PaymentOrderInfo/PaymentOrderInfo';

import {IAviaBookingOrderPageData} from 'server/services/AviaBookingService/order/IAviaBookingOrderPageData';

interface IBookingPaymentInfoProps extends IWithClassName {
    order: IAviaBookingOrderPageData | null | undefined;
}

function prepareDate(date: string | undefined): string | null {
    if (!date) {
        return null;
    }

    return moment(date, ROBOT).format(HUMAN);
}

const BookingPaymentInfo: React.FC<IBookingPaymentInfoProps> = props => {
    const {className, order} = props;

    const dates = useMemo(() => {
        const arrivalDate = order?.segments?.[0]?.flights?.[0]?.departureDate;
        const departureDate = order?.segments?.[1]?.flights?.[0]?.departureDate;

        return (
            <span key="dates">
                {prepareDate(arrivalDate)}
                {departureDate && (
                    <>
                        {' – '}
                        {prepareDate(departureDate)}
                    </>
                )}
            </span>
        );
    }, [order]);

    const settlements: React.ReactNode = useMemo(() => {
        const segments = order?.segments;

        if (!segments) {
            return null;
        }

        const settlementTitles = _flatMap(segments, segment => {
            const {flights} = segment;

            const firstFlight = _head(flights);
            const lastFlight = _last(flights);

            return [
                ...(firstFlight ? [firstFlight.departureSettlement.title] : []),
                ...(lastFlight ? [lastFlight.arrivalSettlement.title] : []),
            ];
        }).reduce((acc, settlement) => {
            const last = _last(acc);

            if (last === settlement) {
                return acc;
            }

            return [...acc, settlement];
        }, [] as string[]);

        return (
            <span key="settlements">
                {settlementTitles.join(`${CHAR_NBSP}– `)}
            </span>
        );
    }, [order]);

    if (!order) {
        return null;
    }

    return (
        <PaymentOrderInfo
            className={className}
            title={insertJSXIntoKey(i18nBlock.paymentDashAmountDashStyleDash2)({
                price: (
                    <Price
                        key="order-price"
                        isRound={false}
                        value={order.price.value}
                        currency={order.price.currency}
                    />
                ),
            })}
            details={[[dates, settlements]]}
        />
    );
};

export default BookingPaymentInfo;
