import React, {PureComponent, ReactNode} from 'react';

import {EProjectName} from 'constants/common';

import {EOrderApiType} from 'server/api/OrdersAPI/enums/types';
import {EFooterProject} from 'components/Footer/types';
import {EAviaOrderDisplayState} from 'types/avia/book/IAviaOrder';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {IWithDeviceType} from 'types/withDeviceType';
import {EAviaOrderError} from 'server/api/OrdersAPI/types/IAviaOrderResponse';

import {getOrderUrl} from 'projects/account/utilities/urls/getOrderUrl';
import {params, reachGoal} from 'utilities/metrika';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlockAviaPlus from 'i18n/avia-plusPromo2021';

import HappyPageLayout from 'components/Layouts/HappyPageLayout/HappyPageLayout';
import AviaHappyPageErrorModal from './components/AviaHappyPageErrorModal/AviaHappyPageErrorModal';
import AviaOrderSegments from './components/AviaOrderSegments/AviaOrderSegments';
import OrderActions, {
    EOrderActionsSource,
} from 'components/OrderActions/OrderActions';
import ErrorPage from 'projects/happyPage/components/ErrorPage/ErrorPage';
import PlusBanner from 'projects/happyPage/components/PlusBanner/PlusBanner';
import AviaHappyPagePaymentErrorModal from 'projects/avia/pages/AviaHappyPage/components/AviaHappyPagePaymentErrorModal/AviaHappyPagePaymentErrorModal';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import HappyPageContent from 'src/projects/happyPage/components/HappyPageContent/HappyPageContent';

import {TAviaHappyPagePropsContainer} from './AviaHappyPageContainer';

import cx from './AviaHappyPage.scss';

const DELAY_BEFORE_FETCH_HAPPY_PAGE_INFO = 5000;

interface IAviaHappyPageProps
    extends IWithDeviceType,
        TAviaHappyPagePropsContainer {}

interface IAviaHappyPageState {
    hasTimeoutBeforeFetchOrder: boolean;
}

class AviaHappyPage extends PureComponent<
    IAviaHappyPageProps,
    IAviaHappyPageState
> {
    state = {
        hasTimeoutBeforeFetchOrder: false,
    };

    qa = 'aviaHappyPage';

    private _repeatFetchHappyPageInfoTimeout?: NodeJS.Timeout;

    componentDidMount(): void {
        this.fetchHappyPageInfoIfCan();

        reachGoal(EAviaGoal.HAPPY_PAGE_SHOW_PAGE);
    }

    componentDidUpdate(prevProps: IAviaHappyPageProps): void {
        const {
            happyPageInfo: {isLoading},
            setIsNewTrip,
        } = this.props;
        const isFulfilledOrderState = this.checkFulfilledOrderState();

        if (
            prevProps.happyPageInfo.isLoading &&
            !isLoading &&
            !isFulfilledOrderState
        ) {
            this.createRepeatFetchHappyPageInfoTimeout();
        }

        if (
            !prevProps.happyPageInfo.value &&
            this.props.happyPageInfo.value &&
            isFulfilledOrderState
        ) {
            setIsNewTrip(true);
        }
    }

    componentWillUnmount(): void {
        this.clearRepeatFetchHappyPageInfoTimeout();
    }

    handleOrderMoreClick(): void {
        reachGoal(EAviaGoal.HAPPY_PAGE_ORDER_MORE_CLICK);
    }

    handleAllHotelsClick(): void {
        reachGoal(EAviaGoal.HAPPY_PAGE_HOTELS_ALL_CLICK);
    }

    handleHotelsMount(): void {
        reachGoal(EAviaGoal.HAPPY_PAGE_SHOW_HOTELS);
    }

    handleOrderInfoMount(): void {
        reachGoal(EAviaGoal.HAPPY_PAGE_SHOW_ORDER_INFO);
    }

    handleErrorMount(): void {
        reachGoal(EAviaGoal.HAPPY_PAGE_SHOW_ERROR_MODAL);
    }

    handlePaymentErrorMount(): void {
        reachGoal(EAviaGoal.HAPPY_PAGE_SHOW_PAYMENT_ERROR_MODAL);
    }

    handlePaymentErrorPrimaryAction(): void {
        reachGoal(EAviaGoal.HAPPY_PAGE_PAYMENT_ERROR_MODAL_SUBMIT);
        params({avia: {withPaymentErrorModal: 1}});
    }

    handlePromoCodeBlockMount(): void {
        reachGoal(EAviaGoal.HAPPY_PAGE_SHOW_PROMO_CODE);
    }

    handlePromoCodeBlockClick(): void {
        reachGoal(EAviaGoal.HAPPY_PAGE_CLICK_PROMO_CODE);
    }

    /* Helpers */

    fetchHappyPageInfoIfCan = (): void => {
        const {fetchHappyPageInfo} = this.props;
        const orderIdByLocation = this.getOrderIdByLocation();

        if (orderIdByLocation) {
            fetchHappyPageInfo({
                orderId: orderIdByLocation,
                orderType: EOrderApiType.AVIA,
            });
        }
    };

    createRepeatFetchHappyPageInfoTimeout = (): void => {
        if (!this._repeatFetchHappyPageInfoTimeout) {
            this._repeatFetchHappyPageInfoTimeout = setTimeout(() => {
                this.fetchHappyPageInfoIfCan();
                this.setState({hasTimeoutBeforeFetchOrder: false});
            }, DELAY_BEFORE_FETCH_HAPPY_PAGE_INFO);

            this.setState({hasTimeoutBeforeFetchOrder: true});
        }
    };

    clearRepeatFetchHappyPageInfoTimeout = (): void => {
        if (this._repeatFetchHappyPageInfoTimeout) {
            clearTimeout(this._repeatFetchHappyPageInfoTimeout);
        }
    };

    checkFulfilledOrderState = (): boolean => {
        const {happyPageInfo} = this.props;
        const displayState = happyPageInfo?.value?.order?.displayState;

        return displayState === EAviaOrderDisplayState.OS_FULFILLED;
    };

    getOrderIdByLocation = (): string | undefined => {
        const {match} = this.props;
        const {
            params: {orderId},
        } = match;

        if (orderId) {
            return orderId;
        }

        return undefined;
    };

    /* Render */

    renderOrderLoader(): ReactNode {
        return (
            <HappyPageLayout
                orderType={EProjectName.AVIA}
                footerType={EFooterProject.AVIA}
                isLoading
            />
        );
    }

    renderErrorModal(): ReactNode {
        const {happyPageInfo} = this.props;
        const pnr = happyPageInfo?.value?.order?.airReservation?.pnr;

        return (
            <AviaHappyPageErrorModal
                pnr={pnr}
                onMount={this.handleErrorMount}
                {...prepareQaAttributes({
                    parent: this.qa,
                    current: 'errorModal',
                })}
            />
        );
    }

    renderPaymentErrorModal(): ReactNode {
        return (
            <AviaHappyPagePaymentErrorModal
                onMount={this.handlePaymentErrorMount}
                onPrimaryAction={this.handlePaymentErrorPrimaryAction}
                {...prepareQaAttributes({
                    parent: this.qa,
                    current: 'paymentErrorModal',
                })}
            />
        );
    }

    renderOrderError(): ReactNode {
        const {
            happyPageInfo: {errorResponse},
        } = this.props;

        const orderId = this.getOrderIdByLocation();

        return (
            <ErrorPage
                statusCode={errorResponse?.status}
                orderType={EProjectName.AVIA}
                footerType={EFooterProject.AVIA}
                orderId={orderId}
                onErrorModalMount={this.handleErrorMount}
                {...prepareQaAttributes({
                    parent: this.qa,
                    current: 'errorPage',
                })}
            />
        );
    }

    renderFlightInfo(): ReactNode {
        return (
            <CardWithDeviceLayout>
                {this.renderOrderSegments()}
            </CardWithDeviceLayout>
        );
    }

    renderBanner(): ReactNode {
        const {
            aviaAeroflotPlus,
            happyPageInfo: {value},
        } = this.props;

        if (
            !aviaAeroflotPlus ||
            !value?.order.promoCampaigns?.plusPromo2021?.enabled
        ) {
            return null;
        }

        return (
            <PlusBanner
                text={i18nBlockAviaPlus.happyPagePromocodeText()}
                deviceType={this.props.deviceType}
                className={cx(deviceMods('plusBanner', this.props.deviceType))}
            />
        );
    }

    renderOrderActions(orderId: string): ReactNode {
        return (
            <OrderActions
                source={EOrderActionsSource.HAPPY_PAGE}
                orderDetails={{
                    url: getOrderUrl(orderId),
                    onClick: this.handleOrderMoreClick,
                }}
                {...prepareQaAttributes({
                    parent: this.qa,
                    current: 'orderActions',
                })}
            />
        );
    }

    renderOrderSegments(): ReactNode {
        const {
            happyPageInfo: {value},
        } = this.props;
        const segments = value?.order?.airReservation.segments;

        if (!segments || segments.length === 0) {
            return null;
        }

        return (
            <AviaOrderSegments
                segments={segments}
                onMount={this.handleOrderInfoMount}
            />
        );
    }

    renderOrder(): ReactNode {
        const {
            happyPageInfo: {value},
        } = this.props;

        if (!value) {
            return null;
        }

        const {
            order: {id, prettyId, email},
            crossSale,
        } = value;

        return (
            <HappyPageLayout
                orderType={EProjectName.AVIA}
                footerType={EFooterProject.AVIA}
            >
                <HappyPageContent
                    orderType={EProjectName.AVIA}
                    orderInfo={this.renderFlightInfo()}
                    banner={this.renderBanner()}
                    orderActions={this.renderOrderActions(id)}
                    crossSale={crossSale}
                    prettyOrderId={prettyId}
                    email={email}
                    crossSaleProps={{
                        onHotelsMount: this.handleHotelsMount,
                        onAllHotelsClick: this.handleAllHotelsClick,
                        onPromoCodeBlockMount: this.handlePromoCodeBlockMount,
                        onPromoCodeBlockClick: this.handlePromoCodeBlockClick,
                    }}
                    {...prepareQaAttributes(this.qa)}
                />
            </HappyPageLayout>
        );
    }

    render(): ReactNode {
        const {
            happyPageInfo: {isLoading, isFailed},
        } = this.props;
        const {hasTimeoutBeforeFetchOrder} = this.state;
        const isFulfilledOrderState = this.checkFulfilledOrderState();
        const isTokenizationFailed =
            this.props.happyPageInfo.value?.order.errorCode ===
            EAviaOrderError.TOKENIZATION_FAILED;

        if (hasTimeoutBeforeFetchOrder || isLoading) {
            return this.renderOrderLoader();
        }

        if (isFulfilledOrderState) {
            return this.renderOrder();
        }

        if (isFailed) {
            return this.renderOrderError();
        }

        if (isTokenizationFailed) {
            return this.renderPaymentErrorModal();
        }

        if (this._repeatFetchHappyPageInfoTimeout) {
            return this.renderErrorModal();
        }

        return this.renderOrderLoader();
    }
}

export default React.memo(AviaHappyPage);
