import {ReactElement, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IAviaOrderSegment} from 'types/avia/book/IAviaOrderSegment';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import AirlineLogos from 'projects/avia/components/AirlineLogos/AirlineLogos';

import cx from './AviaOrderAirlineInfo.scss';

export interface IAviaOrderAirlineInfoProps extends IWithClassName {
    segment: IAviaOrderSegment;
}

const AviaOrderAirlineInfo = (
    props: IAviaOrderAirlineInfoProps,
): ReactElement | null => {
    const {className, segment} = props;
    const {flights} = segment;
    const deviceType = useDeviceType();
    const {isMobile} = deviceType;

    const logos = useMemo(
        () =>
            flights.map(flight => ({
                svg: flight.operatingCompany.logoSvg,
                title: flight.operatingCompany.title,
            })),
        [flights],
    );

    return (
        <Flex
            className={cx('root', className)}
            alignItems="flex-start"
            between={2}
            inline
        >
            <AirlineLogos items={logos} size={isMobile ? 's' : 'm'} />

            <Flex flexDirection="column">
                {flights.map((flight, index) => {
                    const {
                        operatingCompany: {title, iataCode},
                        flightNumber,
                    } = flight;
                    const fullFlightNumber = `${iataCode} ${Number(
                        flightNumber,
                    )}`;

                    return (
                        <div key={index}>
                            <Text
                                className={cx('title')}
                                size={isMobile ? 'm' : 'xl'}
                                weight={isMobile ? 'bold' : 'normal'}
                                color="primary"
                            >
                                {title}
                            </Text>
                            <Text
                                size={isMobile ? 'm' : 'xl'}
                                color="secondary"
                            >
                                {fullFlightNumber}
                            </Text>
                        </div>
                    );
                })}
            </Flex>
        </Flex>
    );
};

export default AviaOrderAirlineInfo;
