import {ReactElement} from 'react';

import {IAviaOrderFlight} from 'types/avia/book/IAviaOrderFlight';
import {IWithClassName} from 'types/withClassName';

import {isNighttimeTransfer} from 'projects/avia/lib/isNighttimeTransfer';
import {hasAirportChange} from 'projects/avia/lib/book/hasAirportChange';
import {getTransferDurationInMinutes} from 'projects/avia/lib/book/getTransferDurationInMinutes';
import {CHAR_EM_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';
import {getDuration} from 'projects/avia/lib/date/getDuration';

import * as i18nAviaDurationBlock from 'i18n/avia-Duration';
import * as i18nBlock from 'i18n/avia-AviaBooking-TransferInfo';

import NightConnectionIcon from 'icons/12/NightConnection';
import ConnectionIcon from 'icons/12/Connection';
import {Duration} from 'projects/avia/components/Duration/Duration';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import Text from 'components/Text/Text';

import cx from './AviaOrderFlightsTransferInfo.scss';

interface IAviaOrderFlightsTransferInfoProps extends IWithClassName {
    departureFlight: IAviaOrderFlight;
    arrivalFlight: IAviaOrderFlight;
}

const AviaOrderFlightsTransferInfo = (
    props: IAviaOrderFlightsTransferInfoProps,
): ReactElement => {
    const {departureFlight, arrivalFlight, className} = props;
    const transferIsNighttime = isNighttimeTransfer({
        departureDate: departureFlight.departureDate,
        arrivalDate: arrivalFlight.arrivalDate,
    });
    const transferWithAirportChange = hasAirportChange(
        arrivalFlight,
        departureFlight,
    );
    const settlement = arrivalFlight.arrivalSettlement.phraseIn;
    const duration = Duration.minutes(
        getTransferDurationInMinutes(arrivalFlight, departureFlight),
    );
    const durationText = i18nAviaDurationBlock
        .dDashHDashMDashFormat(getDuration(duration.get()))
        .trim();

    const durationNode = (
        <span className={cx('duration')}>
            {CHAR_NBSP}
            {CHAR_EM_DASH}
            {CHAR_NBSP}
            {durationText}
        </span>
    );

    if (transferWithAirportChange) {
        return (
            <div className={cx(className)}>
                <TextWithIcon
                    className={className}
                    iconLeft={ConnectionIcon}
                    iconLeftClassName={cx('connectionIcon')}
                    size="m"
                    text={
                        <>
                            {transferIsNighttime
                                ? i18nBlock.nightDashTransferDashInDashWithDashAirportDashChange(
                                      {settlement},
                                  )
                                : i18nBlock.transferDashInDashWithDashAirportDashChange(
                                      {
                                          settlement,
                                      },
                                  )}
                            {durationNode}
                        </>
                    }
                />
            </div>
        );
    }

    return (
        <div className={cx(className)}>
            {transferIsNighttime ? (
                <TextWithIcon
                    iconLeft={NightConnectionIcon}
                    iconSize={12}
                    size="m"
                    text={
                        <>
                            {i18nBlock.nightDashTransferDashIn({settlement})}
                            {durationNode}
                        </>
                    }
                />
            ) : (
                <Text size="m">
                    {i18nBlock.transferDashIn({settlement})}
                    {durationNode}
                </Text>
            )}
        </div>
    );
};

export default AviaOrderFlightsTransferInfo;
