import {ReactElement} from 'react';
import _first from 'lodash/first';
import _last from 'lodash/last';

import {IAviaOrderSegment} from 'types/avia/book/IAviaOrderSegment';
import {IWithClassName} from 'types/withClassName';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import {TwoColumnLayout} from 'components/Layouts/TwoColumnLayout/TwoColumnLayout';
import {AviaFlightTimings} from 'projects/avia/components/AviaFlightTimings/AviaFlightTimings';

import AviaOrderSegmentTitle from '../AviaOrderSegmentTitle/AviaOrderSegmentTitle';
import AviaOrderAirlineInfo from '../AviaOrderAirlineInfo/AviaOrderAirlineInfo';
import AviaOrderSegmentSettlements from '../AviaOrderSegmentSettlements/AviaOrderSegmentSettlements';
import AviaOrderSegmentTransferInfo from '../AviaOrderSegmentTransferInfo/AviaOrderSegmentTransferInfo';

import cx from './AviaOrderSegment.scss';

interface IAviaOrderSegmentProps extends IWithClassName {
    segment: IAviaOrderSegment;
}

const AviaOrderSegment = (
    props: IAviaOrderSegmentProps,
): ReactElement | null => {
    const {segment, className} = props;
    const {flights} = segment;
    const firstSegmentFlight = _first(flights);
    const lastSegmentFlight = _last(flights);
    const deviceType = useDeviceType();

    if (firstSegmentFlight && lastSegmentFlight) {
        return (
            <div className={cx(deviceMods('root', deviceType), className)}>
                <AviaOrderSegmentTitle
                    className={cx('segmentTitle')}
                    segment={segment}
                />
                <TwoColumnLayout
                    deviceType={deviceType}
                    rightColumnWidth={89}
                    rightColumnOffset={5}
                >
                    <TwoColumnLayout.LeftColumn
                        className={cx('airlineInfoColumn')}
                    >
                        <AviaOrderAirlineInfo segment={segment} />
                    </TwoColumnLayout.LeftColumn>
                    <TwoColumnLayout.RightColumn>
                        <AviaFlightTimings
                            arrivalDate={lastSegmentFlight.arrivalDate}
                            departureDate={firstSegmentFlight.departureDate}
                            arrivalAirportTimezoneOffset={
                                lastSegmentFlight.arrivalAirport.timeZoneOffset
                            }
                            departureAirportTimezoneOffset={
                                firstSegmentFlight.departureAirport
                                    .timeZoneOffset
                            }
                        />
                        <AviaOrderSegmentSettlements
                            className={cx('settlements')}
                            firstSegmentFlight={firstSegmentFlight}
                            lastSegmentFlight={lastSegmentFlight}
                        />
                        <AviaOrderSegmentTransferInfo
                            className={cx('transferInfo')}
                            flights={flights}
                        />
                    </TwoColumnLayout.RightColumn>
                </TwoColumnLayout>
            </div>
        );
    }

    return null;
};

export default AviaOrderSegment;
