import React, {useCallback, useEffect, useMemo, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {RouteComponentProps} from 'react-router';

import {EProjectName} from 'constants/common';

import {ETextBlocksType} from 'types/common/seoPages/ITextBlocks';
import {EFooterProject} from 'components/Footer/types';
import {
    ELandingBlockType,
    IClosestCitiesBlock,
    ICityDirectionsBlock,
    ILandingSectionTextBlock,
} from 'types/avia/landing/IAviaLanding';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';

import {requestFlightsToDataThunkAction} from 'reducers/avia/flightsTo/thunk/requestFlightsToDataThunkAction';
import {IAviaRequestFlightsToDataAction} from 'reducers/avia/flightsTo/actions';

import flightsToDataSelector from 'selectors/avia/landing/flightsToDataSelector';

import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {
    deviceModDesktop,
    deviceModMobile,
    deviceMods,
} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import scrollTo from 'utilities/dom/scrollTo';
import {prefetchData} from './utilities/prefetchData';
import {prepareCityDirectionsItems} from 'projects/avia/pages/AviaLanding/utilities/prepareCityDirectionsItems';
import {prepareClosesCitiesItems} from 'projects/avia/pages/AviaLanding/utilities/prepareClosestCitiesItems';

import LinksList from 'components/LinksList/LinksList';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import {TwoColumnLayout} from 'components/Layouts/TwoColumnLayout/TwoColumnLayout';
import Breadcrumbs from 'projects/avia/pages/AviaLanding/components/Breadcrumbs/Breadcrumbs';
import SearchForm from 'projects/avia/pages/AviaLanding/components/SearchForm/SearchForm';
import DynamicsBlock from 'projects/avia/pages/AviaLanding/components/DynamicsBlock/DynamicsBlock';
import LandingSectionTextBlock from 'projects/avia/pages/AviaLanding/components/LandingSectionBlock/LandingSectionBlock';
import Navigation from 'projects/avia/pages/AviaLanding/components/Navigation/Navigation';
import Error from 'projects/avia/pages/AviaLanding/components/Error/Error';
import SchemaMarkup, {
    getFAQSchemaMarkup,
} from 'components/SchemaMarkup/SchemaMarkup';
import BookLoader from 'components/BookLoader/BookLoader';

import cx from './AviaFlightsTo.scss';

const RIGHT_COLUMN_OFFSET = 10;
const RIGHT_COLUMN_WIDTH = 80;

interface IAviaFlightsToProps
    extends RouteComponentProps<IAviaRequestFlightsToDataAction> {}

const AviaFlightsTo: React.FC<IAviaFlightsToProps> = ({match}) => {
    const deviceType = useDeviceType();
    const dispatch = useDispatch();
    const needToRender = useServerDataFetcher([prefetchData(match.params)]);
    const {
        breadcrumbs,
        searchForm,
        dynamicsBlock,
        blocks,
        navigation,
        isError,
        isNotFound,
        isFetched,
        seoInfo,
        nonce,
        aviaContext,
    } = useSelector(flightsToDataSelector);

    const {toSlug} = match.params;

    const needRenderDynamics = Boolean(
        aviaContext.fromId && aviaContext.toId && aviaContext.when,
    );

    const [refsArray, setRef] = useState<HTMLDivElement[]>([]);

    const addRef = useCallback(
        (element: HTMLDivElement) => {
            refsArray.push(element);
            setRef(refsArray);
        },
        [refsArray, setRef],
    );

    const scrollToBlock = useCallback(
        (index: number) => (): void => {
            const top = refsArray[index].offsetTop;

            scrollTo({top});
        },
        [refsArray],
    );

    useEffect(() => {
        dispatch(
            requestFlightsToDataThunkAction({
                toSlug,
            }),
        );
    }, [dispatch, toSlug]);

    const contentBlock = useMemo(() => {
        if (
            !isFetched ||
            !searchForm ||
            !breadcrumbs ||
            !dynamicsBlock ||
            !blocks
        ) {
            return <BookLoader isLoading />;
        }

        const renderBlock = (
            block:
                | ILandingSectionTextBlock
                | IClosestCitiesBlock
                | ICityDirectionsBlock,
            index: number,
        ): React.ReactNode => {
            let getRef: ((elem: HTMLDivElement) => void) | undefined;

            if (block.data.navigationTitle) {
                getRef = addRef;
            }

            switch (block.type) {
                case ETextBlocksType.sectionTextBlock:
                    return (
                        <LandingSectionTextBlock
                            key={index}
                            ref={getRef}
                            className={cx(deviceMods('block', deviceType))}
                            block={block}
                        />
                    );
                case ELandingBlockType.closestCitiesBlock:
                    return (
                        <LinksList
                            key="closestCity"
                            ref={getRef}
                            className={cx(deviceMods('block', deviceType))}
                            title={block.data.title}
                            description={block.data.description}
                            links={prepareClosesCitiesItems(block.data.items)}
                        />
                    );
                case ELandingBlockType.cityDirectionsBlock:
                    return (
                        <LinksList
                            key="cityDirections"
                            ref={getRef}
                            className={cx(deviceMods('block', deviceType))}
                            title={block.data.title}
                            description={block.data.text}
                            links={prepareCityDirectionsItems(
                                block.data.items,
                                toSlug,
                            )}
                        />
                    );
                default:
                    return null;
            }
        };

        return (
            <>
                {seoInfo && (
                    <SchemaMarkup
                        data={getFAQSchemaMarkup(seoInfo.schemaOrg.faqItems)}
                        nonce={nonce}
                    />
                )}

                <SearchForm
                    className={cx(deviceMods('searchFormBlock', deviceType))}
                    block={searchForm}
                />

                {deviceType.isDesktop && (
                    <Breadcrumbs
                        className={cx(
                            'breadcrumbs',
                            deviceModDesktop('breadcrumbs', deviceType),
                        )}
                        block={breadcrumbs}
                    />
                )}

                <div
                    className={cx('content', deviceMods('content', deviceType))}
                >
                    {needRenderDynamics && (
                        <DynamicsBlock
                            className={cx(deviceMods('block', deviceType))}
                            deviceType={deviceType}
                            block={dynamicsBlock}
                            getRef={addRef}
                        />
                    )}

                    <AdFoxBanner
                        type={EAdFoxBannerType.Inline}
                        position={EAdFoxBannerPosition.Center}
                    />
                </div>
                <TwoColumnLayout
                    className={cx('layout', deviceMods('layout', deviceType))}
                    tagName="section"
                    deviceType={deviceType}
                    rightColumnOffset={RIGHT_COLUMN_OFFSET}
                    rightColumnWidth={RIGHT_COLUMN_WIDTH}
                >
                    <TwoColumnLayout.LeftColumn className={cx('left')}>
                        {blocks.map(renderBlock)}
                    </TwoColumnLayout.LeftColumn>
                    <TwoColumnLayout.RightColumn className={cx('right')}>
                        <Navigation
                            deviceType={deviceType}
                            items={navigation || []}
                            onClick={scrollToBlock}
                        />
                    </TwoColumnLayout.RightColumn>
                </TwoColumnLayout>

                {deviceType.isMobile && (
                    <Breadcrumbs
                        className={cx(
                            'breadcrumbs',
                            deviceModMobile('breadcrumbs', deviceType),
                        )}
                        block={breadcrumbs}
                    />
                )}
            </>
        );
    }, [
        addRef,
        blocks,
        breadcrumbs,
        deviceType,
        dynamicsBlock,
        isFetched,
        navigation,
        needRenderDynamics,
        nonce,
        scrollToBlock,
        searchForm,
        seoInfo,
        toSlug,
    ]);

    if (!needToRender) {
        return null;
    }

    if (isError) {
        return <Error isNotFound={isNotFound} />;
    }

    return (
        <LayoutDefault
            isFixedNavigation
            showNavigation
            showHeader
            searchFormInitialIsExpanded
            showFooter
            project={EProjectName.AVIA}
            footerType={EFooterProject.AVIA}
        >
            {contentBlock}
        </LayoutDefault>
    );
};

export default AviaFlightsTo;
