import React, {
    ReactNode,
    useCallback,
    useEffect,
    useMemo,
    useState,
} from 'react';
import {RouteComponentProps} from 'react-router-dom';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';

import {ETextBlocksType} from 'types/common/seoPages/ITextBlocks';
import {EFooterProject} from 'components/Footer/types';
import {
    ELandingBlockType,
    IAviaCompaniesBlock,
    IClosestCitiesBlock,
    ILandingSectionTextBlock,
    IReturnTicketBlock,
} from 'types/avia/landing/IAviaLanding';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {ITransportRoutesBlock} from 'types/common/seoPages/ITransportRoutesBlock';
import {ECommonLandingBlockType} from 'types/common/seoPages/ECommonLandingBlockType';
import {ICrossSaleHotelsBlock} from 'types/common/seoPages/ICrossSaleHotelsBlock';

import {IAviaRequestRouteDataAction} from 'reducers/avia/route/actions';
import {requestRouteDataThunkAction} from 'reducers/avia/route/thunk/requestRouteDataThunkAction';

import routeDataSelector from 'selectors/avia/landing/routeDataSelector';

import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {
    deviceModDesktop,
    deviceModMobile,
    deviceMods,
} from 'utilities/stylesUtils';
import scrollTo from 'utilities/dom/scrollTo';
import {prefetchData} from './utilities/prefetchData';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {prepareReturnTicketItems} from 'projects/avia/pages/AviaLanding/utilities/prepareReturnTicketItems';
import {prepareClosesCitiesItems} from 'projects/avia/pages/AviaLanding/utilities/prepareClosestCitiesItems';
import {prepareAviaCompaniesItems} from 'projects/avia/pages/AviaLanding/utilities/prepareAviaCompaniesItems';

import {hotelsCrossSaleTitle} from 'i18n/avia-route';

import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import TwoColumnLayout from 'components/Layouts/TwoColumnLayout/TwoColumnLayout';
import Breadcrumbs from 'projects/avia/pages/AviaLanding/components/Breadcrumbs/Breadcrumbs';
import SearchForm from 'projects/avia/pages/AviaLanding/components/SearchForm/SearchForm';
import DynamicsBlock from 'projects/avia/pages/AviaLanding/components/DynamicsBlock/DynamicsBlock';
import LandingSectionTextBlock from 'projects/avia/pages/AviaLanding/components/LandingSectionBlock/LandingSectionBlock';
import Navigation from 'projects/avia/pages/AviaLanding/components/Navigation/Navigation';
import Error from 'projects/avia/pages/AviaLanding/components/Error/Error';
import SchemaMarkup, {
    getFAQSchemaMarkup,
} from 'components/SchemaMarkup/SchemaMarkup';
import BookLoader from 'components/BookLoader/BookLoader';
import LinksList from 'components/LinksList/LinksList';
import CrossSaleHotelsBlock from 'components/CrossSaleHotelsBlock/CrossSaleHotelsBlock';
import TransportRoutesBlock from 'components/TransportRoutesBlock/TransportRoutesBlock';
import SearchDynamicLandingWrapper from 'projects/avia/components/Dynamics/SearchDynamic/SearchDynamicLandingWrapper';

import {AviaFiltersBufferProvider} from 'projects/avia/context/AviaFiltersBufferContext';

import cx from './AviaRoute.scss';

const RIGHT_COLUMN_OFFSET = 10;
const RIGHT_COLUMN_WIDTH = 80;
const ROOT_QA = 'avia-directionPage';

interface IAviaRouteProps
    extends RouteComponentProps<IAviaRequestRouteDataAction> {}

const AviaRoute: React.FC<IAviaRouteProps> = ({match}) => {
    const deviceType = useDeviceType();
    const dispatch = useDispatch();
    const needToRender = useServerDataFetcher([prefetchData(match.params)]);

    const {
        breadcrumbs,
        searchForm,
        dynamicsBlock,
        blocks,
        navigation,
        isError,
        isNotFound,
        isFetched,
        seoInfo,
        nonce,
    } = useSelector(routeDataSelector);

    const [refsArray, setRef] = useState<HTMLDivElement[]>([]);

    const addRef = useCallback(
        (element: HTMLDivElement) => {
            refsArray.push(element);
            setRef(refsArray);
        },
        [refsArray, setRef],
    );

    const scrollToBlock = useCallback(
        (index: number) => (): void => {
            const top = refsArray[index].offsetTop;

            scrollTo({top});
        },
        [refsArray],
    );

    const handleDynamicToSearch = useReachGoal(
        EAviaGoal.ROUTES_DYNAMIC_TO_SEARCH,
    );

    useEffect(() => {
        dispatch(
            requestRouteDataThunkAction({
                fromSlug: match.params.fromSlug,
                toSlug: match.params.toSlug,
            }),
        );
    }, [dispatch, match.params.fromSlug, match.params.toSlug]);

    const contentBlock = useMemo(() => {
        if (
            !isFetched ||
            !searchForm ||
            !breadcrumbs ||
            !dynamicsBlock ||
            !blocks
        ) {
            return <BookLoader isLoading />;
        }

        const renderBlock = (
            block:
                | ILandingSectionTextBlock
                | IAviaCompaniesBlock
                | IClosestCitiesBlock
                | ITransportRoutesBlock
                | ICrossSaleHotelsBlock
                | IReturnTicketBlock,
            index: number,
        ): ReactNode => {
            let getRef: ((elem: HTMLDivElement) => void) | undefined;

            if (block.data.navigationTitle) {
                getRef = addRef;
            }

            const commonProps = {
                key: index,
                ref: getRef,
                className: cx(deviceMods('block', deviceType)),
            };

            switch (block.type) {
                case ETextBlocksType.sectionTextBlock:
                    return (
                        <LandingSectionTextBlock
                            {...commonProps}
                            block={block}
                        />
                    );
                case ELandingBlockType.aviaCompaniesBlock:
                    return (
                        <LinksList
                            {...commonProps}
                            title={block.data.title}
                            description={block.data.description}
                            links={prepareAviaCompaniesItems(
                                block.data.items,
                                searchForm.data.searchFormParams.fromId,
                                searchForm.data.searchFormParams.toId,
                            )}
                            priceAsLink
                        />
                    );
                case ELandingBlockType.closestCitiesBlock:
                    return (
                        <LinksList
                            {...commonProps}
                            title={block.data.title}
                            description={block.data.description}
                            links={prepareClosesCitiesItems(block.data.items)}
                        />
                    );
                case ELandingBlockType.returnTicketBlock:
                    return (
                        <LinksList
                            {...commonProps}
                            title={block.data.title}
                            description={block.data.description}
                            links={prepareReturnTicketItems(block.data.items)}
                        />
                    );
                case ECommonLandingBlockType.TRANSPORT_ROUTES:
                    return (
                        <TransportRoutesBlock
                            {...commonProps}
                            data={block.data}
                            vertical={EProjectName.AVIA}
                        />
                    );
                case ECommonLandingBlockType.HOTELS_CROSS_SALE:
                    return (
                        <CrossSaleHotelsBlock
                            title={hotelsCrossSaleTitle({
                                regionName:
                                    block.data.region.linguistics
                                        .prepositionalCase,
                            })}
                            {...commonProps}
                            {...block.data}
                            vertical={EProjectName.AVIA}
                            linkType="region"
                        />
                    );

                default:
                    return null;
            }
        };

        return (
            <>
                {seoInfo && (
                    <SchemaMarkup
                        data={getFAQSchemaMarkup(seoInfo.schemaOrg.faqItems)}
                        nonce={nonce}
                    />
                )}

                <SearchForm
                    className={cx(deviceMods('searchFormBlock', deviceType))}
                    block={searchForm}
                />

                {deviceType.isDesktop && (
                    <Breadcrumbs
                        className={cx(
                            'breadcrumbs',
                            deviceModDesktop('breadcrumbs', deviceType),
                        )}
                        block={breadcrumbs}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'breadcrumps',
                        })}
                    />
                )}

                <div
                    className={cx('content', deviceMods('content', deviceType))}
                >
                    {deviceType.isMobile ? (
                        <SearchDynamicLandingWrapper
                            className={cx(deviceMods('block', deviceType))}
                            onLinkClick={handleDynamicToSearch}
                            isVisible
                        />
                    ) : (
                        <DynamicsBlock
                            className={cx(deviceMods('block', deviceType))}
                            deviceType={deviceType}
                            block={dynamicsBlock}
                            getRef={addRef}
                        />
                    )}

                    <AdFoxBanner
                        type={EAdFoxBannerType.Inline}
                        position={EAdFoxBannerPosition.Center}
                    />
                </div>
                <TwoColumnLayout
                    className={cx('layout', deviceMods('layout', deviceType))}
                    tagName="section"
                    deviceType={deviceType}
                    rightColumnOffset={RIGHT_COLUMN_OFFSET}
                    rightColumnWidth={RIGHT_COLUMN_WIDTH}
                >
                    <TwoColumnLayout.LeftColumn className={cx('left')}>
                        {blocks.map(renderBlock)}
                    </TwoColumnLayout.LeftColumn>
                    <TwoColumnLayout.RightColumn className={cx('right')}>
                        <Navigation
                            deviceType={deviceType}
                            items={navigation || []}
                            onClick={scrollToBlock}
                        />
                    </TwoColumnLayout.RightColumn>
                </TwoColumnLayout>

                {deviceType.isMobile && (
                    <Breadcrumbs
                        className={cx(
                            'breadcrumbs',
                            deviceModMobile('breadcrumbs', deviceType),
                        )}
                        block={breadcrumbs}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'breadcrumps',
                        })}
                    />
                )}
            </>
        );
    }, [
        addRef,
        blocks,
        breadcrumbs,
        deviceType,
        dynamicsBlock,
        isFetched,
        navigation,
        scrollToBlock,
        searchForm,
        nonce,
        seoInfo,
        handleDynamicToSearch,
    ]);

    if (!needToRender) {
        return null;
    }

    if (isError) {
        return <Error isNotFound={isNotFound} />;
    }

    return (
        <AviaFiltersBufferProvider>
            <LayoutDefault
                isFixedNavigation
                showNavigation
                showHeader
                searchFormInitialIsExpanded
                showFooter
                project={EProjectName.AVIA}
                footerType={EFooterProject.AVIA}
            >
                {contentBlock}
            </LayoutDefault>
        </AviaFiltersBufferProvider>
    );
};

export default AviaRoute;
