import React, {useCallback} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {IRouteDynamicsBlock} from 'types/avia/landing/route/IAviaRoute';
import {IFlightsToDynamicsBlock} from 'types/avia/landing/flightsTo/IAviaFlightsTo';
import {EDynamicsDaysMode} from 'projects/avia/components/Dynamics/types/IDynamicsDay';

import titleSelector from 'projects/avia/pages/AviaLanding/components/DynamicsBlock/selectors/titleSelector';

import {reachGoal} from 'utilities/metrika';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {useDidMountEffect} from 'utilities/hooks/useDidMountEffect';

import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import Heading from 'components/Heading/Heading';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import AviaDynamicsCalendarContent from 'projects/avia/components/Dynamics/StandaloneDynamic/StandaloneDynamic';
import DynamicsCommonContainer from 'projects/avia/components/Dynamics/Chart/DynamicsCommonContainer/DynamicsCommonContainer';

import cx from './DynamicsBlock.scss';

interface IDynamicsBlockProps extends IWithClassName, IWithDeviceType {
    block: IRouteDynamicsBlock | IFlightsToDynamicsBlock;
    getRef?: (element: HTMLDivElement) => void;
}

const DynamicsBlock: React.FC<IDynamicsBlockProps> = ({
    className,
    deviceType: {isDesktop},
    block: {data},
    getRef,
}) => {
    const isComponentMounted = useDidMountEffect();

    const title = useSelector(titleSelector);
    const {organicAviaDynamicWeek} = useExperiments();

    const handleSearch = useCallback(() => {
        reachGoal(EAviaGoal.ROUTES_DYNAMIC_TO_SEARCH);
    }, []);

    const handleFilterChange = useCallback(() => {
        reachGoal(EAviaGoal.ROUTES_DYNAMIC_FILTERS);
    }, []);

    const handleScroll = useCallback(() => {
        reachGoal(EAviaGoal.ROUTES_DYNAMIC_MORE_RESULTS);
    }, []);

    const handleModeChange = useCallback((mode: EDynamicsDaysMode) => {
        if (mode === EDynamicsDaysMode.MONTH) {
            reachGoal(EAviaGoal.ROUTES_DYNAMIC_MONTH_SEARCH);
        }
    }, []);

    return (
        <div className={cx(className)} ref={getRef}>
            <Box below={isDesktop ? 5 : 4}>
                <Heading level={2}>{title || data.title}</Heading>
            </Box>
            <Text className={cx('description')} size="m">
                {data.description}
            </Text>
            {isDesktop ? (
                isComponentMounted && (
                    <CardWithDeviceLayout above={3}>
                        <DynamicsCommonContainer
                            view="yak"
                            standalone
                            className={cx('dynamics')}
                            doNotPreventLinkClick
                            onSearch={handleSearch}
                            onFilterChange={handleFilterChange}
                            onScroll={handleScroll}
                            onModeChange={handleModeChange}
                            mode={
                                organicAviaDynamicWeek
                                    ? EDynamicsDaysMode.WEEK
                                    : EDynamicsDaysMode.MONTH
                            }
                        />
                    </CardWithDeviceLayout>
                )
            ) : (
                <Box above={3}>
                    <AviaDynamicsCalendarContent
                        wrapperClassName={cx('mobileDynamics')}
                        onSearch={handleSearch}
                        onFilterChange={handleFilterChange}
                    />
                </Box>
            )}
        </div>
    );
};

export default React.memo(DynamicsBlock);
