import React from 'react';
import {useParams} from 'react-router-dom';
import {useSelector} from 'react-redux';

import {IAviaPointsData} from 'reducers/avia/data-types';

import routeDataSelector from 'selectors/avia/landing/routeDataSelector';
import {getAviaPointsData} from 'selectors/avia/aviaSelectors';

import {aviaURLs} from 'projects/avia/lib/urls';

import SocialSharingMeta from 'components/SocialSharingMeta/SocialSharingMeta';
import DocumentMeta, {
    IAlternateLink,
} from 'components/DocumentMeta/DocumentMeta';

interface IDirectionSlugs {
    fromSlug: string;
    toSlug: string;
}

function getAlternates(
    pointsData: IAviaPointsData,
    {fromSlug, toSlug}: IDirectionSlugs,
): IAlternateLink[] {
    const iataFrom = pointsData.from?.settlement?.iata?.toLowerCase();
    const urlTitleFrom = pointsData.from?.settlement?.urlTitle?.toLowerCase();
    const iataTo = pointsData.to?.settlement?.iata?.toLowerCase();
    const urlTitleTo = pointsData.to?.settlement?.urlTitle?.toLowerCase();

    const hasData = iataFrom && iataTo && urlTitleFrom && urlTitleTo;
    const foreignDomains = hasData
        ? [
              {
                  href: `https://avia.yandex.ua/routes/${iataFrom}/${iataTo}/${urlTitleFrom}-${urlTitleTo}/`,
                  hreflang: 'uk',
              },
              {
                  href: `https://avia.yandex.ua/routes/${iataFrom}/${iataTo}/${urlTitleFrom}-${urlTitleTo}/?lang=ru`,
                  hreflang: 'ru-UA',
              },
              {
                  href: `https://avia.yandex.ua/routes/${iataFrom}/${iataTo}/${urlTitleFrom}-${urlTitleTo}/?lang=en`,
                  hreflang: 'en-UA',
              },
              {
                  href: `https://bilet.yandex.com.tr/routes/${iataFrom}/${iataTo}/${urlTitleFrom}-${urlTitleTo}/`,
                  hreflang: 'tr',
              },
              {
                  href: `https://bilet.yandex.com.tr/routes/${iataFrom}/${iataTo}/${urlTitleFrom}-${urlTitleTo}/?lang=en`,
                  hreflang: 'en-TR',
              },
              {
                  href: `https://avia.yandex.kz/routes/${iataFrom}/${iataTo}/${urlTitleFrom}-${urlTitleTo}/`,
                  hreflang: 'ru-KZ',
              },
          ]
        : [];

    return [
        {
            href: aviaURLs.getRoutePageUrl(fromSlug, toSlug, undefined, {
                withOrigin: true,
            }),
            hreflang: 'ru',
        },
        ...foreignDomains,
    ];
}

const RouteMetaHelmet: React.FC = () => {
    const {seoInfo} = useSelector(routeDataSelector);
    const pointsData = useSelector(getAviaPointsData);
    const directionSlugs = useParams<IDirectionSlugs>();
    const {fromSlug, toSlug} = directionSlugs;

    const absoluteRouteUrl = aviaURLs.getRoutePageUrl(
        fromSlug,
        toSlug,
        undefined,
        {withOrigin: true},
    );

    if (!seoInfo) {
        return null;
    }

    const {
        title,
        description,
        openGraph: {
            title: ogTitle,
            description: ogDescription,
            image: ogImage,
            imageSize: ogImageSize,
        },
    } = seoInfo;

    return (
        <>
            <DocumentMeta
                title={title}
                description={description}
                canonicalUrl={absoluteRouteUrl}
                alternates={getAlternates(pointsData, directionSlugs)}
            />
            <SocialSharingMeta
                url={absoluteRouteUrl}
                title={ogTitle}
                description={ogDescription}
                image={ogImage}
                imageHeight={ogImageSize?.height ?? ''}
                imageWidth={ogImageSize?.width ?? ''}
                twitterCardType="summary_large_image"
            />
        </>
    );
};

export default RouteMetaHelmet;
