import React, {useEffect} from 'react';
import {connect, useDispatch, useSelector} from 'react-redux';
import {compose} from 'redux';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {EAviaUserSearchLogPages} from 'server/loggers/avia/AviaUserSearchLog/types/AVIA_USER_SEARCH_LOG_PAGES';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {EAviaActionLogPageName} from 'server/loggers/avia/AviaActionLog/types/EAviaActionLogPageName';

import {withPrevState} from 'reducers/utils/withPrevState';
import {setAviaCurrentPage} from 'reducers/avia/page/actions';
import {aviaOrderActions} from 'reducers/avia/order/actions';

import aviaOrderSelector, {
    aviaOrderExtendedSelector,
    aviaOrderSelectors,
} from 'selectors/avia/order/aviaOrderSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {reachGoal} from 'utilities/metrika';
import {EAviaRumEvents} from 'projects/avia/lib/EAviaRumEvents';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import AviaLayout from 'projects/avia/components/AviaLayout';
import AviaError500 from 'projects/avia/components/AviaError500';

import useRequestOrder from 'projects/avia/pages/AviaOrder/hooks/useRequestOrder';

import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';
import {useRumUi, withRumUiSubpage} from 'contexts/RumUiContext';
import {useCoordinator} from 'contexts/PlatformContext';

import {prefillAviaPointsData} from 'server/redux/avia/prefillAviaPointsData';
import {prefetchAviaOrder} from 'server/redux/avia/prefetchOrder';
import {prefillAviaContext} from 'server/redux/avia/prefillSearchForm';

import {aviaLoggingBrowserProvider} from 'serviceProvider/avia/logging/aviaLoggingBrowserProvider';

import {AviaOrderContent} from './AviaOrderContent';
import useOffers from './hooks/useOffers';
import useVariants from './hooks/useVariants';

export interface IAviaOrderRouteProps {
    variantKey?: string;
}

export interface IAviaOrderOwnProps extends IAviaOrderRouteProps {
    isModalView: boolean;
}

type TAviaOrderProps = ReturnType<typeof aviaOrderSelector> &
    ReturnType<typeof aviaOrderExtendedSelector> &
    IAviaOrderOwnProps;

const ROOT_QA = 'aviaOrder';

const AviaOrder: React.FC<TAviaOrderProps> = ({
    qid,
    context,
    useOrderData,
    hasGoodPrice,
    searchIsCompleted,
    progress,
    offers: rawOffers,
    baggagePrice,
    reference,
    error,
    isModalView,
    defaultPlusPoints,
    cheapestVariant,
    sortedVariants,
}) => {
    const dispatch = useDispatch();
    const badPartners = useSelector(aviaOrderSelectors.badPartners);
    const deviceType = useDeviceType();
    const totalSeats =
        Number(context.adult_seats) +
        Number(context.children_seats) +
        Number(context.infant_seats);
    const rumUi = useRumUi();

    const {
        variants,
        selectedVariant,
        handleSelect: onSelectVariant,
    } = useVariants(rawOffers?.variantsByTariff);

    const offers = useOffers(rawOffers, variants);

    const coordinator = useCoordinator();

    const isSearchInProgress = Boolean(qid) || Boolean(cheapestVariant);

    useRequestOrder(Boolean(cheapestVariant?.route.length));

    useEffect(() => {
        dispatch(
            withPrevState(setAviaCurrentPage(EAviaActionLogPageName.ORDER)),
        );

        return () => {
            dispatch(aviaOrderActions.stopDataFetching());
        };
    }, [dispatch]);

    useEffect(() => {
        if (useOrderData) {
            rumUi.measure(EAviaRumEvents.OrderViaDirectUrl);
        }
    }, [dispatch, useOrderData, qid, rumUi, isSearchInProgress]);

    useEffect(() => {
        if (qid) {
            aviaLoggingBrowserProvider.userSearch(
                qid,
                EAviaUserSearchLogPages.ORDER,
            );
            reachGoal(EAviaGoal.ORDER_SHOW_PAGE);
            coordinator.doAction(EAppActions.REPORT_METRICS_EVENT, {
                name: EAviaGoal.ORDER_SHOW_PAGE,
            });
        }
    }, [coordinator, qid]);

    useEffect(() => {
        if (badPartners.length) {
            reachGoal(EAviaGoal.ORDER_SHOW_PAGE_AFTER_REDIRECT_ERROR, {
                avia: {
                    badPartners: badPartners.reduce<Record<string, boolean>>(
                        (acc, partner) => {
                            acc[partner] = true;

                            return acc;
                        },
                        {},
                    ),
                },
            });
        }
    }, [badPartners]);

    if (error) {
        return <AviaError500 />;
    }

    const content = (
        <AviaOrderContent
            offers={offers}
            deviceType={deviceType}
            hasGoodPrice={hasGoodPrice}
            badPartners={badPartners}
            baggagePrice={baggagePrice}
            context={context}
            reference={reference}
            searchIsCompleted={searchIsCompleted}
            totalSeats={totalSeats}
            qid={qid ?? undefined}
            progress={progress}
            isModalView={isModalView}
            defaultPlusPoints={defaultPlusPoints}
            cheapestVariant={cheapestVariant}
            selectedVariant={selectedVariant}
            onSelectVariant={onSelectVariant}
            sortedVariants={sortedVariants}
            {...prepareQaAttributes(ROOT_QA)}
        />
    );

    return deviceType.isMobile ? (
        content
    ) : (
        <AviaLayout {...prepareQaAttributes(ROOT_QA)}>{content}</AviaLayout>
    );
};

export default compose(
    serverFetchDataDispatcher(
        [prefetchAviaOrder, prefillAviaContext, prefillAviaPointsData],
        {checkNested: true},
    ),
    connect(aviaOrderSelector),
    connect(aviaOrderExtendedSelector),
    withRumUiSubpage('avia-order'),
)(AviaOrder);
