import React, {useMemo, useState} from 'react';
import {useSelector} from 'react-redux';

import {IOS} from 'server/constants/platforms';

import {IProgress} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {emptyFareFamilyKeys} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {IWithDeviceType} from 'types/withDeviceType';

import {IAviaContext} from 'reducers/avia/data-types';
import {INormalizedTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';

import {aviaAeroflotPlusSelector} from 'projects/avia/selectors/aviaAeroflotPlusEnabledSelector';
import {IAviaOrderOffers} from 'selectors/avia/order/aviaOrderSelector';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {EAviaRumEvents} from 'projects/avia/lib/EAviaRumEvents';
import IPrice from 'utilities/currency/PriceInterface';
import {isOS} from 'utilities/deviceType/isOS';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlockAviaPlus from 'i18n/avia-plusPromo2021';

import {AviaSearchProgressBar} from 'projects/avia/components/AviaSearchProgressBar/AviaSearchProgressBar';
import PlusInfo from 'projects/avia/components/PlusInfo/PlusInfo';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import AviaOrderHeader from './components/AviaOrderHeader/AviaOrderHeader';
import AviaOrderMeta from './components/AviaOrderMeta/AviaOrderMeta';
import BestOffer from './components/BestOffer/BestOffer';
import AviaFlightPlaceholder from './components/FlightPlaceholder';
import OrderSubscriptionModal from './components/OrderSubscriptionModal/OrderSubscriptionModal';
import PriceListBlock from './components/PriceListBlock/PriceListBlock';
import AviaOrderFlightList from 'projects/avia/pages/AviaOrder/components/OrderRoute';
import Card from 'components/Card/Card';
import AviaOrderDisclaimers from 'projects/avia/pages/AviaOrder/components/Disclaimers/AviaOrderDisclaimers';
import useUniqVariants from 'projects/avia/components/AviaResultVariant/components/TariffSelector/hooks/useUniqVariants';
import TariffsBottomSheet from 'projects/avia/pages/AviaOrder/components/TariffsBottomSheet/TariffsBottomSheet';

import ScopeContext from 'contexts/ScopeContext';
import {useRumObserveState} from 'contexts/RumUiContext';
import {usePlatform} from 'contexts/PlatformContext';

import cx from './AviaOrder.scss';

interface IAviaOrderContentProps extends IWithDeviceType, IWithQaAttributes {
    offers: IAviaOrderOffers | null;
    sortedVariants: IResultAviaVariant[];
    badPartners: string[];
    hasGoodPrice: boolean;
    baggagePrice: IPrice | null;
    context: IAviaContext;
    reference: INormalizedTDReference;
    searchIsCompleted: boolean;
    totalSeats: number;
    qid?: string;
    isModalView: boolean;
    progress: IProgress | null;
    defaultPlusPoints?: number;
    cheapestVariant?: IResultAviaVariant;
    selectedVariant: IResultAviaVariant | null;
    onSelectVariant(variantToSelect: IResultAviaVariant): void;
}

export const AviaOrderContent: React.FC<IAviaOrderContentProps> = props => {
    const {
        offers,
        deviceType,
        hasGoodPrice,
        badPartners,
        baggagePrice,
        context,
        reference,
        searchIsCompleted,
        totalSeats,
        qid,
        progress,
        isModalView,
        defaultPlusPoints,
        cheapestVariant,
        selectedVariant,
        onSelectVariant,
        sortedVariants,
    } = props;

    useRumObserveState(EAviaRumEvents.OrderViaSerp, [offers]);
    useRumObserveState(EAviaRumEvents.OrderViaDirectUrl, [offers]);

    const {isMobile} = deviceType;
    const rootQa = getQa(props);

    const aviaAeroflotPlus = useSelector(aviaAeroflotPlusSelector);

    const {isTravelApp} = usePlatform();

    const variants = useUniqVariants(
        offers?.variantsByTariff.withoutBaggage.noOrUnknownRefund,
        offers?.variantsByTariff.withoutBaggage.chargedRefund,
        offers?.variantsByTariff.withoutBaggage.freeRefund,
        offers?.variantsByTariff.withBaggage.noOrUnknownRefund,
        offers?.variantsByTariff.withBaggage.chargedRefund,
        offers?.variantsByTariff.withBaggage.freeRefund,
    );

    const plusInfoBlock = useMemo(() => {
        if (!aviaAeroflotPlus || !defaultPlusPoints) {
            return null;
        }

        return (
            <PlusInfo
                className={cx('plusInfo', deviceMods('plusInfo', deviceType))}
                plusPoints={defaultPlusPoints}
                deviceType={deviceType}
                popupText={i18nBlockAviaPlus.plusPopupVariantText()}
                tankerTranslation={
                    deviceType.isMobile
                        ? i18nBlockAviaPlus.variantsPromoRefundMobile
                        : i18nBlockAviaPlus.variantsPromoRefundDesktop
                }
            />
        );
    }, [aviaAeroflotPlus, deviceType, defaultPlusPoints]);

    const [container, setContainer] = useState<HTMLDivElement | null>(null);

    const flightList = useMemo(() => {
        const isRoundtrip = Boolean(selectedVariant?.backwardNumber);

        const emptyRoute = !cheapestVariant || !cheapestVariant.route.length;

        if (emptyRoute)
            return <AviaFlightPlaceholder isRoundtrip={isRoundtrip} />;

        const fareFamilies =
            selectedVariant?.price.fareFamilies ?? emptyFareFamilyKeys;

        return cheapestVariant.route.map((flights, index) => (
            <AviaOrderFlightList
                key={index}
                context={context}
                flights={flights}
                variant={cheapestVariant}
                hasBaggage={cheapestVariant.hasBaggage}
                baggage={cheapestVariant.price.baggage[index]}
                fareFamilyKeys={fareFamilies[index]}
                reference={reference}
                selectedVariant={selectedVariant || cheapestVariant}
                isLastItem={index === cheapestVariant.route.length - 1}
                {...prepareQaAttributes({
                    parent: rootQa,
                    current: `${index ? 'backward' : 'forward'}-route`,
                })}
            />
        ));
    }, [cheapestVariant, context, reference, rootQa, selectedVariant]);

    container?.addEventListener('scroll', () => {
        console.log('scroll');
    });

    return (
        <>
            <AviaOrderMeta variant={selectedVariant} offers={offers} />

            <div
                {...prepareQaAttributes(
                    `${rootQa}-${searchIsCompleted ? 'complete' : 'searching'}`,
                )}
                className={cx('root', deviceMods('root', deviceType))}
                ref={setContainer}
            >
                <ScopeContext.Provider value={container || undefined}>
                    <AviaOrderHeader
                        useBackButton={isModalView}
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'header',
                        })}
                    />

                    <div
                        className={cx('body', deviceMods('body', deviceType), {
                            body_withoutMarginBottom: isTravelApp,
                        })}
                    >
                        {!isTravelApp && (
                            <AdFoxBanner
                                className={cx(
                                    'banner',
                                    deviceMods('banner', deviceType),
                                )}
                                type={EAdFoxBannerType.Inline}
                                position={EAdFoxBannerPosition.Top}
                            />
                        )}

                        <div className={cx('content')}>
                            <Card
                                x={4}
                                y={4}
                                below={3}
                                shadow="none"
                                className={cx(
                                    'routesWrapper',
                                    deviceMods('routesWrapper', deviceType),
                                )}
                            >
                                {flightList}
                            </Card>

                            {!isTravelApp ? (
                                <div className={cx('priceList')}>
                                    <PriceListBlock
                                        baggagePrice={baggagePrice}
                                        error={!offers && searchIsCompleted}
                                        partnersError={badPartners.length !== 0}
                                        context={context}
                                        hasGoodPrice={hasGoodPrice}
                                        offers={offers}
                                        qid={qid}
                                        totalSeats={totalSeats}
                                        plusInfoBlock={plusInfoBlock}
                                        selectedVariant={selectedVariant}
                                        onSelectVariant={onSelectVariant}
                                        {...prepareQaAttributes({
                                            parent: rootQa,
                                            current: 'offers',
                                        })}
                                    />
                                </div>
                            ) : (
                                <>
                                    {isTravelApp &&
                                        selectedVariant &&
                                        variants.length && (
                                            <TariffsBottomSheet
                                                offers={sortedVariants}
                                                variants={variants}
                                                selectedVariant={
                                                    selectedVariant
                                                }
                                                onSelect={onSelectVariant}
                                            />
                                        )}
                                    <AviaOrderDisclaimers />
                                </>
                            )}
                        </div>
                    </div>

                    <div className={cx('fixedFooter')}>
                        <AviaSearchProgressBar
                            visible={!searchIsCompleted}
                            all={progress ? progress.all : 0}
                            current={progress ? progress.current : 0}
                            fixed={false}
                        />

                        {isMobile && offers?.cheapest && qid && !isTravelApp && (
                            <BestOffer
                                className={cx('footerBestOffer', {
                                    footerBestOffer_ios: isOS(
                                        deviceType.os,
                                        IOS,
                                    ),
                                })}
                                offers={offers}
                                qid={qid}
                                selectedVariant={selectedVariant}
                            />
                        )}
                    </div>
                </ScopeContext.Provider>
            </div>

            <OrderSubscriptionModal />
        </>
    );
};
