import React, {memo, MouseEventHandler, useCallback} from 'react';
import {useLocation} from 'react-router-dom';
import {parse} from 'query-string';
import {omit} from 'lodash';

import {EAppActions} from 'constants/platforms/TPlatforms';

import history from 'utilities/browserHistory/browserHistory';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {aviaURLs} from 'projects/avia/lib/urls';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/avia-order';

import Link from 'components/Link/Link';
import {Icon} from 'components/Icon/Icon';
import BackArrowIcon from 'icons/16/ArrowBack';
import Box from 'components/Box/Box';
import {ShareHorizontal} from 'components/ShareHorizontal/ShareHorizontal';

import {useCoordinator, usePlatform} from 'contexts/PlatformContext';

import cx from './AviaOrderHeader.scss';

interface IAviaOrderHeaderProps extends IWithQaAttributes {
    useBackButton: boolean;
}

// Пропускаем эти параметры при формировании ссылки на поиск, т.к. они относятся
// к логике страницы покупки
const PARAMS_BLACKLIST = ['fromError', 'partner', 'baggage'];

const AviaOrderHeader: React.FC<IAviaOrderHeaderProps> = props => {
    const {useBackButton} = props;
    const deviceType = useDeviceType();
    const {isMobile} = deviceType;
    const {search, hash} = useLocation();
    const searchUrl = aviaURLs.getSearchResultsUrl(
        omit(parse(search), PARAMS_BLACKLIST),
        parse(hash),
    );
    const {isTravelApp} = usePlatform();
    const coordinator = useCoordinator();
    const clickHandler: MouseEventHandler<HTMLElement> = useCallback(
        e => {
            if (isTravelApp) {
                coordinator.doAction(EAppActions.NAVIGATE_TO_SEARCH_AVIA_PAGE);
                e.preventDefault();

                return;
            }

            if (useBackButton && history) {
                e.preventDefault();
                history.goBack();
            }
        },
        [useBackButton, isTravelApp, coordinator],
    );

    return (
        <div
            className={cx(
                'subheader-wrapper',
                deviceMods('subheader-wrapper', deviceType),
            )}
        >
            <Box
                className={cx('subheader', deviceMods('subheader', deviceType))}
                textSize="m"
            >
                <Link
                    to={searchUrl}
                    onClick={clickHandler}
                    theme="black"
                    innerRef={(node: HTMLAnchorElement | null): void => {
                        if (node && node.href !== searchUrl) {
                            node.href = searchUrl;
                        }
                    }}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'backLink',
                    })}
                >
                    <Icon>
                        <BackArrowIcon />
                    </Icon>
                    {i18nBlock.backDashButtonDashTitle()}
                </Link>

                {!isMobile && (
                    <ShareHorizontal
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'shareBlock',
                        })}
                    />
                )}
            </Box>
        </div>
    );
};

export default memo(AviaOrderHeader);
