/* eslint-disable camelcase */
import {connect} from 'react-redux';
import {stringify as qsStringify} from 'query-string';

import {PRODUCTION_PROJECT_URL} from 'constants/common';

import {IAviaContext, IAviaPointsData} from 'reducers/avia/data-types';

import {getAviaContext, getAviaPointsData} from 'selectors/avia/aviaSelectors';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {IAviaOrderOffers} from 'selectors/avia/order/aviaOrderSelector';

import {humanFormatDate} from 'utilities/dateUtils';
import {formatPrice} from 'utilities/currency';
import getTextFunctions from 'projects/avia/pages/AviaOrder/components/AviaOrderMeta/utilities/getTextFunctions';

import DocumentMeta from 'components/DocumentMeta/DocumentMeta';
import SocialSharingMeta from 'components/SocialSharingMeta/SocialSharingMeta';

interface IAviaOrderMetaOwnProps {
    variant: IResultAviaVariant | null;
    offers: IAviaOrderOffers | null;
}

interface IAviaOrderMetaStateProps {
    aviaPointsData: IAviaPointsData;
    aviaContext: IAviaContext;
}

type IAviaOrderMetaProps = IAviaOrderMetaStateProps & IAviaOrderMetaOwnProps;

function _AviaOrderMeta({
    aviaPointsData,
    aviaContext,
    variant,
    offers,
}: IAviaOrderMetaProps) {
    const {from, to} = aviaPointsData;
    const {when, return_date, fromName, toName} = aviaContext;

    if (!from || !to || !when || !variant) {
        return null;
    }

    const fromSettlement = from.settlement;
    const toSettlement = to.settlement;

    const humanWhen = humanFormatDate(when);
    const humanReturnDate = return_date && humanFormatDate(return_date);

    const textFunctions = getTextFunctions(
        Boolean(offers),
        Boolean(return_date),
    );

    const metaInfo = {
        ogDescription: textFunctions.ogDescription({
            fromPointPhrase: fromSettlement?.phraseFrom ?? '',
            toPointPhrase: toSettlement?.phraseTo ?? '',
            forwardAirport: variant.route[0][0]?.from?.title ?? '',
            forwardNumber: variant.forwardNumber,
            backwardAirport: variant.route[1][0]?.from?.title ?? '',
            backwardNumber: variant.backwardNumber ?? '',
        }),
        ogTitle: textFunctions.ogTitle({
            fromPointTitle: fromName,
            toPointTitle: toName,
            when: humanWhen,
            returnDate: humanReturnDate,
            minPrice:
                (offers && formatPrice(offers.cheapest.price.tariff)) ?? '',
        }),
        title: textFunctions.title({
            fromPointPhrase: fromName,
            toPointTitle: toName,
            when: humanWhen,
            returnDate: humanReturnDate,
            minPrice:
                (offers && formatPrice(offers.cheapest.price.tariff)) ?? '',
        }),
    };

    const searchQuery = qsStringify({
        fromName,
        toName,
        when,
        return_date,
        forward: variant.forwardRoute,
        backward: variant.backwardRoute,
    });
    const ogUrl = `${PRODUCTION_PROJECT_URL}/avia/order?${searchQuery}`;

    return (
        <>
            <DocumentMeta title={metaInfo.title} />
            <SocialSharingMeta
                url={ogUrl}
                title={metaInfo.ogTitle}
                description={metaInfo.ogDescription}
            />
        </>
    );
}

const mapStateToProps = (state: any) => ({
    aviaPointsData: getAviaPointsData(state),
    aviaContext: getAviaContext(state),
});

export default connect<IAviaOrderMetaStateProps, any, IAviaOrderMetaOwnProps>(
    mapStateToProps,
)(_AviaOrderMeta);
