import React, {FunctionComponent, useCallback} from 'react';

import {EAviaGoal} from 'utilities/metrika/types/goals/avia';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceModMobile} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {EAviaOfferType} from 'projects/avia/lib/search/offerType';

import * as i18nBlock from 'i18n/avia-aviaOrder-Offer';

import Flex from 'components/Flex/Flex';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import BorderPositioned from 'components/BorderPositioned/BorderPositioned';
import PartnerLogo from 'projects/avia/components/PartnerLogo/PartnerLogo';
import OfferTypeHint from 'projects/avia/components/OfferTypeHint/OfferTypeHint';
import AviaVariantPriceWithHint from 'projects/avia/components/AviaVariantPrice/AviaVariantPriceWithHint';
import RedirectLink from 'projects/avia/components/RedirectLink/RedirectLink';
import PlusColoredLabel from 'components/PlusColoredLabel/PlusColoredLabel';

import cx from './AviaOrderOffer.scss';

interface IAviaOrderOfferProps extends IWithQaAttributes {
    offer: IResultAviaVariant;
    isBestOffer?: boolean;
    label?: React.ReactNode;
    offerType?: EAviaOfferType;
    qid: string;
    onClick?: (offer: IResultAviaVariant) => void;
}

const AviaOrderOffer: FunctionComponent<IAviaOrderOfferProps> = props => {
    const {offer, isBestOffer = false, offerType, label, onClick} = props;
    const {price} = offer;
    const {partner, charter, plusPoints} = price;
    const deviceType = useDeviceType();

    const onClickHandler = useCallback(
        () => onClick?.(offer),
        [onClick, offer],
    );

    return (
        <div
            onClick={onClickHandler}
            className={cx('root')}
            {...prepareQaAttributes(props)}
        >
            <PartnerLogo
                className={cx('partner')}
                size={isBestOffer ? 'm' : 's'}
                partner={partner}
                isBoy={price.boy}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'logo',
                })}
            />

            <div className={cx('priceAndButton')}>
                <Flex flexDirection="column" alignItems="flex-end">
                    {label}

                    <AviaVariantPriceWithHint
                        nationalPriceClassName={cx({bestPrice: isBestOffer})}
                        theme="order"
                        price={price}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'price',
                        })}
                    />

                    {offerType && charter && <OfferTypeHint type={offerType} />}
                </Flex>

                <BorderPositioned
                    badge={
                        plusPoints && <PlusColoredLabel count={plusPoints} />
                    }
                    direction="top-right-plus"
                >
                    <ButtonLink
                        className={cx('button')}
                        size="m-inset"
                        theme={isBestOffer ? 'primary' : 'secondary'}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'button',
                        })}
                    >
                        {i18nBlock.buyDashButton()}
                    </ButtonLink>
                </BorderPositioned>
            </div>

            <RedirectLink
                className={cx(
                    'pressedPlace',
                    deviceModMobile('pressedPlace', deviceType),
                )}
                variant={offer}
                goal={EAviaGoal.ORDER_REDIRECT}
            />
        </div>
    );
};

export default AviaOrderOffer;
