import React, {useCallback} from 'react';
import moment from 'moment';

import {IBaggageTDTariff} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {TVariantFareFamilyKey} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';

import {INormalizedTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';

import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {HUMAN_SHORT, ROBOT} from 'utilities/dateUtils/formats';
import {parseDate} from 'utilities/dateUtils';
import {getTypeOfAviaSegment} from 'projects/avia/lib/search/segments';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Box from 'components/Box/Box';
import Card from 'components/Card/Card';
import Flex from 'components/Flex/Flex';
import Separator from 'components/Separator/Separator';
import AirlineLogo from 'projects/avia/components/AirlineLogo/AirlineLogo';
import FlightInfo from 'projects/avia/pages/AviaOrder/components/FlightInfo/FlightInfo';
import {AviaFlightTimings} from 'projects/avia/components/AviaFlightTimings/AviaFlightTimings';
import {PassengerExperience} from 'projects/avia/components/PassengerExperience/PassengerExperience';
import {PassengerExperienceItem} from 'projects/avia/components/PassengerExperience/PassengerExperienceItem';
import {AviaCustomTransportType} from 'projects/avia/components/AviaCustomTransportType/AviaCustomTransportType';

import cx from './Flight.scss';

interface IFlightProps extends IWithQaAttributes {
    flight: IResultAviaFlight;
    baggage: IBaggageTDTariff;
    hasBaggage: boolean;
    isBusiness: boolean;
    previousFlight: IResultAviaFlight;
    reference: INormalizedTDReference;
    className?: string;
    fareFamilyKey: TVariantFareFamilyKey;
    selectedVariant: IResultAviaVariant;
}

interface IRowItem extends IWithQaAttributes {
    content?: React.ReactNode;
}

const Flight: React.FC<IFlightProps> = props => {
    const {
        flight,
        baggage,
        hasBaggage,
        isBusiness,
        previousFlight,
        reference,
        className,
        fareFamilyKey,
        selectedVariant,
    } = props;
    const {isDesktop, isMobile} = useDeviceType();

    const {from, to, number, departure, arrival} = flight;
    const company = flight.operating?.company || flight.company;

    const nextDayDeparture =
        previousFlight &&
        !moment(departure.local).isSame(previousFlight.arrival.local, 'day');
    const nextDayArrival = !moment(departure.local).isSame(
        arrival.local,
        'day',
    );

    const renderRow = useCallback(
        (items: IRowItem[], mod?: string) =>
            items.filter(({content}) => Boolean(content)).length ? (
                <Flex
                    className={cx('dataRow', mod && `dataRow_${mod}`)}
                    inline
                    nowrap
                    between="1"
                    textSize="m"
                >
                    {items.map((item: IRowItem, index: number) => (
                        <div
                            key={`${mod}${index}`}
                            className={cx('dataRowItem')}
                            {...prepareQaAttributes(item)}
                        >
                            {Boolean(item.content) && item.content}
                        </div>
                    ))}
                </Flex>
            ) : null,
        [],
    );

    const [iata, flightNumber] = number.split(' ');

    return (
        <Card
            shadow="default"
            className={cx('root', {isMobile}, className)}
            {...prepareQaAttributes(props)}
        >
            <div
                className={cx('company')}
                style={{backgroundColor: company?.color}}
            >
                <AirlineLogo
                    logoSrc={company.logoSvg}
                    airlineTitle={company.title}
                    size={isMobile ? 'm' : 'l'}
                />
            </div>

            <div className={cx('details')}>
                <div className={cx('main')}>
                    <div className={cx('title')}>
                        {company && company.title && (
                            <span
                                className={cx('companyName')}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'companyTitle',
                                })}
                            >
                                {company.title}
                            </span>
                        )}

                        <span
                            className={cx('plane')}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'planeNumber',
                            })}
                        >
                            {number}
                        </span>
                    </div>

                    <div className={cx('aboutPlane')}>
                        <PassengerExperience
                            when={parseDate(departure.local).format(ROBOT)}
                            iata={iata}
                            flightNumber={flightNumber}
                        >
                            {({
                                isLoading,
                                passengerExperience,
                            }): React.ReactNode => {
                                if (isLoading || !passengerExperience) {
                                    return null;
                                }

                                return (
                                    <Box textSize="s">
                                        <PassengerExperienceItem
                                            field="plane-model"
                                            data={passengerExperience}
                                            {...prepareQaAttributes(props)}
                                        />
                                        <PassengerExperienceItem
                                            field="seat-pitch"
                                            data={passengerExperience}
                                            {...prepareQaAttributes(props)}
                                        />
                                        <PassengerExperienceItem
                                            field="seats-total"
                                            data={passengerExperience}
                                            {...prepareQaAttributes(props)}
                                        />
                                    </Box>
                                );
                            }}
                        </PassengerExperience>

                        <AviaCustomTransportType
                            className={cx('customTransport')}
                            type={getTypeOfAviaSegment(from.tType, to.tType)}
                        />
                    </div>

                    {renderRow(
                        [
                            {
                                content:
                                    nextDayDeparture &&
                                    moment(departure.local).format(HUMAN_SHORT),
                                ...prepareQaAttributes({
                                    parent: props,
                                    current: 'nextDayDeparture',
                                }),
                            },
                            {
                                content:
                                    nextDayArrival &&
                                    moment(arrival.local).format(HUMAN_SHORT),
                                ...prepareQaAttributes({
                                    parent: props,
                                    current: 'nextDayArrival',
                                }),
                            },
                        ],
                        'nextDay',
                    )}

                    <AviaFlightTimings
                        departureDate={departure.local}
                        departureAirportTimezoneOffset={departure.offset}
                        arrivalDate={arrival.local}
                        arrivalAirportTimezoneOffset={arrival.offset}
                        view="order"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'timings',
                        })}
                    />

                    {renderRow(
                        [
                            {
                                content: from.settlement?.title,
                                ...prepareQaAttributes({
                                    parent: props,
                                    current: 'fromCity',
                                }),
                            },
                            {
                                content: to.settlement?.title,
                                ...prepareQaAttributes({
                                    parent: props,
                                    current: 'toCity',
                                }),
                            },
                        ],
                        'city',
                    )}

                    {renderRow(
                        [
                            {
                                content: from.title,
                                ...prepareQaAttributes({
                                    parent: props,
                                    current: 'fromAirport',
                                }),
                            },
                            {
                                content: to.title,
                                ...prepareQaAttributes({
                                    parent: props,
                                    current: 'toAirport',
                                }),
                            },
                        ],
                        'airportTitle',
                    )}

                    {renderRow(
                        [
                            {
                                content: from.code,
                                ...prepareQaAttributes({
                                    parent: props,
                                    current: 'fromAirportCode',
                                }),
                            },
                            {
                                content: to.code,
                                ...prepareQaAttributes({
                                    parent: props,
                                    current: 'toAirportCode',
                                }),
                            },
                        ],
                        'airportCode',
                    )}
                </div>

                {isDesktop && <Separator isVertical height="100%" />}

                <FlightInfo
                    className={cx('additional')}
                    flight={flight}
                    baggage={baggage}
                    hasBaggage={hasBaggage}
                    isBusiness={isBusiness}
                    reference={reference}
                    fareFamilyKey={fareFamilyKey}
                    selectedVariant={selectedVariant}
                    {...prepareQaAttributes(props)}
                />
            </div>
        </Card>
    );
};

export default Flight;
