import {FunctionComponent, ReactNode} from 'react';

import {IWithClassName} from 'types/withClassName';
import {
    IBaggageTDTariff,
    IRating,
    TFlightNumber,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {TVariantFareFamilyKey} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';

import {INormalizedFareFamily} from 'reducers/avia/utils/ticketDaemon/normalizeFareFamilies';
import {INormalizedTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';

import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';
import {buildTariffInfo} from 'selectors/avia/utils/denormalization/tariff';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {
    doesBaggageHaveSizes,
    isBaggageIncluded,
} from 'projects/avia/lib/baggage';
import * as flightDelay from 'projects/avia/lib/reference/flightDelay';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/avia-PassengerExperience';

import Box from 'components/Box/Box';
import {Icon} from 'components/Icon/Icon';
import DelayIcon from 'icons/16/Delay';
import BaggageIcon from 'icons/16/Baggage';
import HandLuggage from 'icons/16/HandLuggage';

import {FlightInfoFareFamily} from '../FlightInfoFareFamily/FlightInfoFareFamily';

import cx from './FlightInfo.scss';

interface IFlightInfoProps extends IWithClassName, IWithQaAttributes {
    flight: IResultAviaFlight;
    baggage: IBaggageTDTariff;
    hasBaggage: boolean;
    isBusiness: boolean;
    reference: INormalizedTDReference;
    fareFamilyKey: TVariantFareFamilyKey;
    selectedVariant: IResultAviaVariant;
}

function renderRaiting(
    number: TFlightNumber,
    ratings: Record<TFlightNumber, IRating>,
): ReactNode {
    const currentRating = ratings && ratings[number];
    const delayInfo =
        flightDelay.hasInfo(currentRating) &&
        flightDelay.getInfo(currentRating);

    if (!delayInfo || delayInfo.intime) {
        return null;
    }

    return (
        <div className={cx('option', 'option_delay', 'option_attention')}>
            <Icon>
                <DelayIcon />
            </Icon>
            <span>{delayInfo.shortText}</span>
        </div>
    );
}

function renderBaggage(
    flight: IResultAviaFlight,
    baggage: IBaggageTDTariff,
    hasBaggage: boolean,
): ReactNode {
    const {aviaCompany} = flight;
    const isHybrid = Boolean(aviaCompany && aviaCompany!.costType === 'hybrid');
    const withSize = doesBaggageHaveSizes(baggage);

    let baggageText = i18nBlock.withoutDashBaggage();
    let paidBaggage = !isBaggageIncluded(baggage);

    if (isHybrid && !hasBaggage) {
        paidBaggage = true;
        baggageText = i18nBlock.maybeDashPaidDashBaggage();
    } else if (hasBaggage || !paidBaggage) {
        baggageText = withSize
            ? i18nBlock.withDashBaggageDashSize({
                  number_pieces: baggage.pc?.count ?? 0,
                  baggage_weight: baggage.wt?.count ?? 0,
              })
            : i18nBlock.withDashBaggage();
    }

    return (
        <div
            className={cx(
                'option',
                'option_baggage',
                paidBaggage && 'option_attention',
            )}
        >
            <Icon className={cx('option__icon')}>
                <BaggageIcon />
            </Icon>
            <span>{baggageText}</span>
        </div>
    );
}

function renderBaggageOptions(
    flight: IResultAviaFlight,
    baggage: IBaggageTDTariff,
    hasBaggage: boolean,
    isBusiness: boolean,
): ReactNode {
    const {short: shortInfo} = buildTariffInfo(
        flight.companyTariff!,
        flight.aviaCompany!,
        isBusiness,
    );

    return (
        <>
            {renderBaggage(flight, baggage, hasBaggage)}
            {shortInfo.map((item, index) => (
                <div
                    key={index}
                    className={cx(
                        'option',
                        `option_${item.type}`,
                        item.status === 'paid' && 'option_attention',
                    )}
                >
                    <Icon className={cx('option__icon')}>
                        {item.type === 'carryon' && <HandLuggage />}
                        {item.type === 'baggage' && <BaggageIcon />}
                    </Icon>
                    <span>{item.text}</span>
                </div>
            ))}
        </>
    );
}

function getFareFamily(
    fareFamilyKey: TVariantFareFamilyKey,
    reference: INormalizedTDReference,
): Nullable<INormalizedFareFamily> {
    return fareFamilyKey ? reference.fareFamilies[fareFamilyKey] : null;
}

const FlightInfo: FunctionComponent<IFlightInfoProps> = ({
    reference,
    baggage,
    hasBaggage,
    isBusiness,
    flight,
    className,
    fareFamilyKey,
    selectedVariant,
    ...rest
}) => {
    const fareFamily = getFareFamily(fareFamilyKey, reference);
    const deviceType = useDeviceType();

    return (
        <Box className={className} between={2}>
            {fareFamily ? (
                <FlightInfoFareFamily
                    fareFamily={fareFamily}
                    deviceType={deviceType}
                    selectedVariant={selectedVariant}
                    {...prepareQaAttributes(rest)}
                />
            ) : (
                <>
                    {renderRaiting(flight.number, reference.ratings)}
                    {renderBaggageOptions(
                        flight,
                        baggage,
                        hasBaggage,
                        isBusiness,
                    )}
                </>
            )}
        </Box>
    );
};

export default FlightInfo;
