import React, {Component} from 'react';

import {IWithDeviceType} from 'types/withDeviceType';

import {
    INormalizedFareFamily,
    INormalizedTerms,
} from 'reducers/avia/utils/ticketDaemon/normalizeFareFamilies';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {TermsHelper} from './utils/TermsHelper';
import {RenderTermsShort} from './utils/RenderTermsShort';
import {RenderTermsExplained} from './utils/RenderTermsExplained';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/avia-AviaOrder-FareFamilies';

import TriggerDetailsButton from 'components/TriggerDetailsButton/TriggerDetailsButton';
import {EMessageBoxPopupTheme} from 'components/MessageBoxPopup/MessageBoxPopup';
import Text from 'components/Text/Text';
import IconArrowButtom from 'icons/12/ArrowBottom';
import TariffDetailsBottomSheet from './components/TariffDetailsBottomSheet/TariffDetailsBottomSheet';

import cx from './FlightInfoFareFamily.scss';

interface IFlightInfoFareFamilyProps
    extends IWithQaAttributes,
        IWithDeviceType {
    fareFamily: INormalizedFareFamily;
    selectedVariant: IResultAviaVariant;
}

interface IFlightInfoFareFamilyState {
    isBottomSheetVisible: boolean;
}

export class FlightInfoFareFamily extends Component<
    IFlightInfoFareFamilyProps,
    IFlightInfoFareFamilyState
> {
    private static renderTermsExplained(
        terms: INormalizedTerms,
    ): React.ReactNode {
        return (
            <ul className={cx('fare-family__options-list-explained')}>
                {RenderTermsExplained.baggage(terms.baggage)}
                {RenderTermsExplained.carryOn(terms.carry_on)}
                {RenderTermsExplained.refundable(terms.refundable)}
                {RenderTermsExplained.changingCarriage(
                    TermsHelper.getAvailability(terms.changing_carriage),
                )}
                {RenderTermsExplained.miles(TermsHelper.getMiles(terms.miles))}
                {RenderTermsExplained.seatSelectionCheckIn(
                    TermsHelper.getAvailability(terms.seat_selection_check_in),
                )}
                {RenderTermsExplained.disclosureUrl(terms.disclosure_url)}
            </ul>
        );
    }

    private static renderTermsShort(
        terms: INormalizedTerms,
        qa: IWithQaAttributes,
    ): React.ReactNode {
        return (
            <ul className={cx('fare-family__options-list-short')}>
                {RenderTermsShort.baggage(terms.baggage, qa)}
                {RenderTermsShort.luggage(terms.carry_on, qa)}
                {RenderTermsShort.miles(TermsHelper.getMiles(terms.miles), qa)}
                {RenderTermsShort.refundable(terms.refundable, qa)}
                {RenderTermsShort.changeCarriage(
                    TermsHelper.getAvailability(terms.changing_carriage),
                    qa,
                )}
                {RenderTermsShort.seatSelectionCheckIn(
                    TermsHelper.getAvailability(terms.seat_selection_check_in),
                    qa,
                )}
            </ul>
        );
    }

    state = {
        isBottomSheetVisible: false,
    };

    private toggleBottomSheet = (): void => {
        this.setState(state => ({
            ...state,
            isBottomSheetVisible: !state.isBottomSheetVisible,
        }));
    };

    render(): React.ReactNode {
        const {fareFamily, deviceType, selectedVariant} = this.props;
        const {isDesktop, isMobile} = deviceType;
        const {isBottomSheetVisible} = this.state;

        return (
            <>
                <div
                    className={cx(
                        'fare-family',
                        deviceMods('fare-family', deviceType),
                    )}
                >
                    <div
                        className={cx('fare-family__tariffName-wrapper')}
                        onClick={this.toggleBottomSheet}
                    >
                        <Text
                            size="s"
                            weight={isDesktop ? 'bold' : 'normal'}
                            className={cx('fare-family__tariffName')}
                            {...prepareQaAttributes({
                                parent: this.props,
                                current: 'tariffName',
                            })}
                        >
                            {i18nBlock.tariffDashTitle({
                                tariffName: fareFamily.tariffGroupName,
                            })}
                            {isMobile && (
                                <IconArrowButtom
                                    className={cx('fare-family__arrowBottom')}
                                />
                            )}
                        </Text>
                        {isDesktop && (
                            <TriggerDetailsButton
                                theme={EMessageBoxPopupTheme.WHITE}
                                {...prepareQaAttributes({
                                    parent: this.props,
                                    current: 'tariffNameInfoTriggerButton',
                                })}
                            >
                                {FlightInfoFareFamily.renderTermsExplained(
                                    fareFamily.terms,
                                )}
                            </TriggerDetailsButton>
                        )}
                    </div>
                    {isDesktop &&
                        FlightInfoFareFamily.renderTermsShort(
                            fareFamily.terms,
                            prepareQaAttributes(this.props),
                        )}
                </div>
                {isMobile && (
                    <TariffDetailsBottomSheet
                        selectedVariant={selectedVariant}
                        fareFamily={fareFamily}
                        isVisible={isBottomSheetVisible}
                        toggleVisible={this.toggleBottomSheet}
                    />
                )}
            </>
        );
    }
}
