import React from 'react';

import {
    IFFBaggageTermRule,
    IFFCarryOnTermRule,
    IFFRefundableTermRule,
    EFareFamilyAvailability,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {isDeprecatedPrice} from 'utilities/currency/IDeprecatedPrice';

import * as i18nBlock from 'i18n/avia-AviaOrder-FareFamilies';

import Link from 'components/Link/Link';
import Price from 'components/Price/Price';

import cx from '../FlightInfoFareFamily.scss';

const withColon = (str: string): string => str + ': ';

const changeCarriageAvailabilityStatusText = {
    [EFareFamilyAvailability.free]:
        i18nBlock.termDashChangingCarriageDashFreeDashExplained(),
    [EFareFamilyAvailability.charge]:
        i18nBlock.termDashChangingCarriageDashPaidDashExplained(),
    [EFareFamilyAvailability.notAvailable]:
        i18nBlock.termDashChangingCarriageDashDeniedDashExplained(),
};

const seatSelectionCheckInAvailabilityStatusText = {
    [EFareFamilyAvailability.free]:
        i18nBlock.termDashSeatSelectionCheckInDashFreeDashExplained(),
    [EFareFamilyAvailability.charge]:
        i18nBlock.termDashSeatSelectionCheckInDashPaidDashExplained(),
    [EFareFamilyAvailability.notAvailable]:
        i18nBlock.termDashSeatSelectionCheckInDashDeniedDashExplained(),
};

export class RenderTermsExplained {
    static baggage(rule?: IFFBaggageTermRule): React.ReactNode {
        if (rule) {
            const hasBaggage = rule.places > 0;

            const title = hasBaggage
                ? withColon(
                      i18nBlock.termDashBaggageDashFreeDashTitleDashExplained(),
                  )
                : i18nBlock.termDashBaggageDashPaidDashTitleDashExplained();

            return (
                <>
                    <li className={cx('fare-family__option-explained')}>
                        <span
                            className={cx(
                                'fare-family__option-explained-title',
                            )}
                        >
                            {title}
                        </span>
                        {hasBaggage && (
                            <span>
                                {RenderTermsExplained.getBaggageDescription(
                                    rule,
                                )}
                            </span>
                        )}
                    </li>
                    {RenderTermsExplained.baggageSize(rule.size)}
                </>
            );
        }

        return null;
    }

    private static baggageSize(size?: string): React.ReactNode {
        if (size) {
            return (
                <li className={cx('fare-family__option-explained')}>
                    <span className={cx('fare-family__option-explained-title')}>
                        {withColon(
                            i18nBlock.baggageDashSizeDashTitleDashExplained(),
                        )}
                    </span>
                    <span>{RenderTermsExplained.getSizeDescription(size)}</span>
                </li>
            );
        }

        return null;
    }

    static carryOn(rule?: IFFCarryOnTermRule): React.ReactNode {
        if (rule) {
            const hasCarryOn = rule.places > 0;

            const title = hasCarryOn
                ? withColon(
                      i18nBlock.termDashLuggageDashFreeDashTitleDashExplained(),
                  )
                : i18nBlock.termDashLuggageDashPaidDashTitleDashExplained();

            return (
                <>
                    <li className={cx('fare-family__option-explained')}>
                        <span
                            className={cx(
                                'fare-family__option-explained-title',
                            )}
                        >
                            {title}
                        </span>
                        {hasCarryOn && (
                            <span>
                                {RenderTermsExplained.getBaggageDescription(
                                    rule,
                                )}
                            </span>
                        )}
                    </li>
                    {RenderTermsExplained.carryOnSize(rule.size)}
                </>
            );
        }

        return null;
    }

    private static carryOnSize(size?: string): React.ReactNode {
        if (size) {
            return (
                <li className={cx('fare-family__option-explained')}>
                    <span className={cx('fare-family__option-explained-title')}>
                        {withColon(
                            i18nBlock.luggageDashSizeDashTitleDashExplained(),
                        )}
                    </span>
                    <span>{RenderTermsExplained.getSizeDescription(size)}</span>
                </li>
            );
        }

        return null;
    }

    static refundable(refundable?: IFFRefundableTermRule): React.ReactNode {
        const refundType = refundable?.availability ?? null;
        const refundPrice = refundable?.charge;

        let refundableRule;

        if (refundType) {
            refundableRule = (
                <li className={cx('fare-family__option-explained')}>
                    <span className={cx('fare-family__option-explained-title')}>
                        {withColon(
                            i18nBlock.termDashRefundableDashTitleDashShort(),
                        )}
                    </span>

                    <span>
                        {refundType === EFareFamilyAvailability.notAvailable &&
                            i18nBlock.termRefundableNotAvailableExplained()}
                        {refundType === EFareFamilyAvailability.charge &&
                            (isDeprecatedPrice(refundPrice)
                                ? insertJSXIntoKey(
                                      i18nBlock.termRefundableChargeCharge,
                                  )({
                                      price: (
                                          <Price
                                              withConversion={false}
                                              {...refundPrice}
                                          />
                                      ),
                                  })
                                : refundable?.comment ??
                                  i18nBlock.termRefundableChargeExplained())}
                        {refundType === EFareFamilyAvailability.free &&
                            i18nBlock.termRefundableFreeExplained()}
                    </span>
                </li>
            );
        }

        return refundableRule;
    }

    static changingCarriage(
        changeCarriage: Nullable<EFareFamilyAvailability>,
    ): React.ReactNode {
        let changingCarriageRule;

        if (changeCarriage) {
            changingCarriageRule = (
                <li className={cx('fare-family__option-explained')}>
                    <span className={cx('fare-family__option-explained-title')}>
                        {withColon(
                            i18nBlock.termDashChangingCarriageDashTitleDashShort(),
                        )}
                    </span>

                    <span>
                        {changeCarriageAvailabilityStatusText[changeCarriage]}
                    </span>
                </li>
            );
        }

        return changingCarriageRule;
    }

    static miles(miles: Nullable<number>): React.ReactNode {
        if (miles === null) {
            return null;
        }

        return (
            <li className={cx('fare-family__option-explained')}>
                {i18nBlock.termDashMilesDashExplained({
                    miles,
                })}
            </li>
        );
    }

    static disclosureUrl(url?: string): React.ReactNode {
        if (url) {
            return (
                <li className={cx('fare-family__option-explained')}>
                    <Link theme="normal" url={url} target="_blank">
                        {i18nBlock.termDashDisclosureUrlDashTitleDashExplained()}
                    </Link>
                </li>
            );
        }

        return null;
    }

    static seatSelectionCheckIn(
        availability: EFareFamilyAvailability | null,
    ): React.ReactNode {
        if (availability === null) {
            return null;
        }

        return (
            <li className={cx('fare-family__option-explained')}>
                {seatSelectionCheckInAvailabilityStatusText[availability]}
            </li>
        );
    }

    private static getSizeDescription(size: string): string {
        // разруливает ситуацию, когда багаж представлен строками вида "158" или "55x40x25", регекспы не хотелось тащить
        if (isNaN(Number(size)) && size.split('x').length) {
            return i18nBlock.sizeDashDescriptionDashExplainedDashTripple({
                size,
            });
        }

        return i18nBlock.sizeDashDescriptionDashExplainedDashSingle({size});
    }

    private static getBaggageDescription({
        places,
        weight,
    }: IFFBaggageTermRule | IFFCarryOnTermRule): string {
        return [
            i18nBlock.placesDashDescriptionDashExplained({
                places,
            }),
            weight &&
                i18nBlock.weightDashDescriptionDashExplained({
                    weight,
                }),
        ]
            .filter(Boolean)
            .join(' ');
    }
}
