import React from 'react';

import {
    EFareFamilyAvailability,
    IFFBaggageTermRule,
    IFFCarryOnTermRule,
    IFFRefundableTermRule,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {isDeprecatedPrice} from 'utilities/currency/IDeprecatedPrice';

import * as i18nBlock from 'i18n/avia-AviaOrder-FareFamilies';

import {Icon} from 'components/Icon/Icon';
import ExcludeIcon from 'icons/16/Exclude';
import IncludeIcon from 'icons/16/Include';
import WarningIcon from 'icons/16/Warning';
import BaggageIcon from 'icons/16/Baggage';
import HandLuggage from 'icons/16/HandLuggage';
import NoTicketReturnIcon from 'icons/16/NoTicketReturn';
import TicketReturnIcon from 'icons/16/TicketReturn';
import Price from 'components/Price/Price';

import cx from '../FlightInfoFareFamily.scss';

const icons = {
    [EFareFamilyAvailability.free]: <IncludeIcon />,
    [EFareFamilyAvailability.charge]: <WarningIcon />,
    [EFareFamilyAvailability.notAvailable]: <ExcludeIcon />,
};

const availabilityStatusText = {
    [EFareFamilyAvailability.free]: (term: string): string =>
        i18nBlock.availabilityDashFreeDashShort({term}),
    [EFareFamilyAvailability.charge]: (term: string): string =>
        i18nBlock.availabilityDashPaidDashShort({term}),
    [EFareFamilyAvailability.notAvailable]: (term: string): string =>
        i18nBlock.availabilityDashDeniedDashShort({term}),
};

const withColon = (str: string): string => str + ': ';

export class RenderTermsShort {
    static baggage(
        rule: IFFBaggageTermRule | undefined,
        qa: IWithQaAttributes,
    ): React.ReactNode {
        if (rule) {
            const noBaggage = rule.places === 0;
            const str = noBaggage
                ? i18nBlock.termDashBaggageDashPaidDashTitleDashShort()
                : RenderTermsShort.getBaggageDescription(rule);

            return (
                <li
                    className={cx(
                        'fare-family__option-short',
                        noBaggage && 'fare-family__option-highlight',
                    )}
                    {...prepareQaAttributes({
                        parent: qa,
                        current: 'baggageInfo',
                    })}
                >
                    <Icon
                        className={cx('fare-family__option-icon')}
                        {...prepareQaAttributes({
                            parent: qa,
                            current: 'baggageIcon',
                        })}
                    >
                        <BaggageIcon />
                    </Icon>
                    <span
                        className={cx('fare-family__option-short-description')}
                        {...prepareQaAttributes({
                            parent: qa,
                            current: 'baggageText',
                        })}
                    >
                        {`${i18nBlock.baggage()}: ${str}`}
                    </span>
                </li>
            );
        }

        return null;
    }

    static luggage(
        rule: IFFCarryOnTermRule | undefined,
        qa: IWithQaAttributes,
    ): React.ReactNode {
        if (rule) {
            const noLuggage = rule.places === 0;
            const str = noLuggage
                ? i18nBlock.termDashLuggageDashPaidDashTitleDashShort()
                : RenderTermsShort.getLuggageDescription(rule);

            return (
                <li
                    className={cx(
                        'fare-family__option-short',
                        noLuggage && 'fare-family__option-highlight',
                    )}
                    {...prepareQaAttributes({
                        parent: qa,
                        current: 'luggageInfo',
                    })}
                >
                    <Icon
                        {...prepareQaAttributes({
                            parent: qa,
                            current: 'luggageIcon',
                        })}
                    >
                        <HandLuggage />
                    </Icon>
                    <span
                        className={cx('fare-family__option-short-description')}
                        {...prepareQaAttributes({
                            parent: qa,
                            current: 'luggageText',
                        })}
                    >
                        {`${i18nBlock.carryOn()}: ${str}`}
                    </span>
                </li>
            );
        }

        return null;
    }

    static miles(
        miles: Nullable<number>,
        qa: IWithQaAttributes,
    ): React.ReactNode {
        if (miles === null) {
            return null;
        }

        const hasMiles = miles > 0;

        return (
            <li
                className={cx('fare-family__option-short')}
                {...prepareQaAttributes({
                    parent: qa,
                    current: 'milesInfo',
                })}
            >
                <Icon
                    {...prepareQaAttributes({
                        parent: qa,
                        current: 'milesIcon',
                    })}
                >
                    {hasMiles ? <IncludeIcon /> : <ExcludeIcon />}
                </Icon>
                <span
                    className={cx('fare-family__option-short-description')}
                    {...prepareQaAttributes({
                        parent: qa,
                        current: 'milesText',
                    })}
                >
                    {hasMiles
                        ? i18nBlock.termDashMilesDashAmountDashShort({miles})
                        : i18nBlock.termDashMilesDashAbsentDashShort()}
                </span>
            </li>
        );
    }

    static refundable(
        refundable?: IFFRefundableTermRule,
        qa?: IWithQaAttributes,
    ): React.ReactNode {
        const refundType = refundable?.availability ?? null;
        const refundPrice = refundable?.charge;

        if (refundable) {
            return (
                <li
                    className={cx('fare-family__option-short')}
                    {...prepareQaAttributes({
                        parent: qa,
                        current: 'refundableInfo',
                    })}
                >
                    <Icon
                        {...prepareQaAttributes({
                            parent: qa,
                            current: 'refundableIcon',
                        })}
                    >
                        {refundType === EFareFamilyAvailability.notAvailable ? (
                            <NoTicketReturnIcon />
                        ) : (
                            <TicketReturnIcon />
                        )}
                    </Icon>
                    <span
                        className={cx('fare-family__option-short-description')}
                        {...prepareQaAttributes({
                            parent: qa,
                            current: 'refundableText',
                        })}
                    >
                        {withColon(
                            i18nBlock.termDashRefundableDashTitleDashShort(),
                        )}
                        {refundType === EFareFamilyAvailability.notAvailable &&
                            i18nBlock.termRefundableNotAvailableExplained()}
                        {refundType === EFareFamilyAvailability.charge &&
                            (isDeprecatedPrice(refundPrice)
                                ? insertJSXIntoKey(
                                      i18nBlock.termRefundableChargeCharge,
                                  )({
                                      price: (
                                          <Price
                                              withConversion={false}
                                              {...refundPrice}
                                          />
                                      ),
                                  })
                                : i18nBlock.termRefundableChargeExplained())}
                        {refundType === EFareFamilyAvailability.free &&
                            i18nBlock.termRefundableFreeExplained()}
                    </span>
                </li>
            );
        }

        return null;
    }

    static changeCarriage(
        changeCarriage: Nullable<EFareFamilyAvailability>,
        qa: IWithQaAttributes,
    ): React.ReactNode {
        return RenderTermsShort.availability(
            i18nBlock.termDashChangingCarriageDashTitleDashShort(),
            changeCarriage,
            prepareQaAttributes({parent: qa, current: 'changeCarriage'}),
        );
    }

    static seatSelectionCheckIn(
        availability: Nullable<EFareFamilyAvailability>,
        qa: IWithQaAttributes,
    ): React.ReactNode {
        return RenderTermsShort.availability(
            i18nBlock.termDashSeatSelectionCheckInDashTitleDashShort(),
            availability,
            prepareQaAttributes({parent: qa, current: 'seatSelectionCheckIn'}),
        );
    }

    private static getBaggageDescription({
        places,
        weight,
    }: IFFBaggageTermRule): React.ReactNode {
        return [
            i18nBlock.baggageDashPlacesDashDescriptionDashShort({
                places,
            }),
            weight &&
                i18nBlock.baggageDashWeightDashDescriptionDashShort({
                    weight,
                }),
        ]
            .filter(Boolean)
            .join(', ');
    }

    private static getLuggageDescription({
        places,
        weight,
        size,
    }: IFFCarryOnTermRule): React.ReactNode {
        const hasSize = Boolean(size);

        return [
            i18nBlock.luggageDashPlacesDashDescriptionDashShort({
                places,
            }),
            weight &&
                i18nBlock.luggageDashWeightDashDescriptionDashShort({
                    weight,
                }),
            hasSize &&
                i18nBlock.luggageDashSizeDashDescriptionDashShort({
                    size,
                }),
        ]
            .filter(Boolean)
            .join(', ');
    }

    private static availability(
        term: string,
        availability: Nullable<EFareFamilyAvailability>,
        qa: IWithQaAttributes,
    ): React.ReactNode {
        if (availability === null) {
            return null;
        }

        const icon = icons[availability];

        const str = availabilityStatusText[availability](term);

        return (
            <div
                className={cx('fare-family__option-short')}
                {...prepareQaAttributes({parent: qa, current: 'info'})}
            >
                <Icon {...prepareQaAttributes({parent: qa, current: 'icon'})}>
                    {icon}
                </Icon>
                <span
                    className={cx('fare-family__option-short-description')}
                    {...prepareQaAttributes({parent: qa, current: 'text'})}
                >
                    {str}
                </span>
            </div>
        );
    }
}
