import React, {useEffect} from 'react';
import {connect} from 'react-redux';
import moment from 'moment-timezone';
import {createSelector} from 'reselect';

import {EAviaGoal} from 'utilities/metrika/types/goals/avia';

import {StoreInterface} from 'reducers/storeTypes';

import {aviaOrderSelectors} from 'selectors/avia/order/aviaOrderSelector';
import {getAviaContext} from 'selectors/avia/aviaSelectors';

import {useMobile} from 'utilities/hooks/useMobile';
import {getNow} from 'utilities/dateUtils';
import {useToggle} from 'utilities/hooks/useToggle';
import {reachGoal} from 'utilities/metrika';
import {decodeQidOrDefault} from 'projects/avia/lib/qid';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {aviaURLs} from 'projects/avia/lib/urls';

import * as i18nBlock from 'i18n/avia-order';

import ModalWithHistoryBack from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import Modal from 'components/Modal/Modal';
import Button from 'components/Button/Button';
import Heading from 'components/Heading/Heading';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import SearchForm from 'projects/avia/components/SearchForm/SearchForm';

import cx from './OfferListError.scss';

interface IAviaOfferListErrorProps extends IWithQaAttributes {
    title: string;
    message: string;
    className?: string;
}

const AviaOfferListBaseError: React.FC<IAviaOfferListErrorProps> = props => {
    const isMobile = useMobile();

    return (
        <div
            {...prepareQaAttributes(props)}
            className={cx('root', isMobile && 'root_mobile', props.className)}
        >
            <Heading level={3} className={cx('title')}>
                {props.title}
            </Heading>

            <div className={cx('message')}>{props.message}</div>

            {props.children}
        </div>
    );
};

interface IAviaOfferListNotFoundErrorProps {
    qid: string;
}

const AviaOfferListNotFoundError: React.FC<IAviaOfferListNotFoundErrorProps> =
    ({qid}) => {
        const searchUrl = aviaURLs.getSearchResultsUrl(decodeQidOrDefault(qid));

        useEffect(() => {
            reachGoal(EAviaGoal.ORDER_NOT_FOUND_ERROR);
        }, []);

        return (
            <AviaOfferListBaseError
                title={i18nBlock.pricesDashNotDashFoundDashTitle()}
                message={i18nBlock.pricesDashNotDashFoundDashText()}
                {...prepareQaAttributes('avia-order-not-found-error')}
            >
                <ButtonLink to={searchUrl} width="max" size="l">
                    {i18nBlock.pricesDashNotDashFoundDashButton()}
                </ButtonLink>
            </AviaOfferListBaseError>
        );
    };

const AviaOfferListOutdateError: React.FC = () => {
    const deviceType = useDeviceType();
    const [formIsVisible, toggleForm] = useToggle(false);

    useEffect(() => {
        reachGoal(EAviaGoal.ORDER_OUTDATE_ERROR);
    }, []);

    return (
        <AviaOfferListBaseError
            title={i18nBlock.pricesDashNotDashFoundDashOutdatedDashTitle()}
            message={i18nBlock.pricesDashNotDashFoundDashOutdatedDashText()}
            {...prepareQaAttributes('aviaOrderOutdateError')}
        >
            {deviceType.isMobile && (
                <>
                    <Button onClick={toggleForm} width="max" size="l">
                        {i18nBlock.pricesDashNotDashFoundDashOutdatedDashButton()}
                    </Button>

                    <ModalWithHistoryBack
                        isMobile
                        fullScreen
                        disableAutoFocus
                        isVisible={formIsVisible}
                        onClose={toggleForm}
                    >
                        <Modal.Content>
                            <SearchForm
                                className={cx('searchForm')}
                                useFilters
                            />
                        </Modal.Content>
                    </ModalWithHistoryBack>
                </>
            )}
        </AviaOfferListBaseError>
    );
};

const outdateSelector = createSelector(
    (state: StoreInterface) => getAviaContext(state).when,
    when => moment(when).isBefore(moment(getNow()), 'day'),
);
const errorSelector = createSelector(
    outdateSelector,
    aviaOrderSelectors.qid,
    (outdate, qid) => ({
        outdate,
        qid,
    }),
);

type TAviaOfferListErrorProps = ReturnType<typeof errorSelector>;

const AviaOfferListError: React.FC<TAviaOfferListErrorProps> = props =>
    props.outdate ? (
        <AviaOfferListOutdateError />
    ) : (
        <AviaOfferListNotFoundError qid={props.qid || ''} />
    );

export default connect(errorSelector)(AviaOfferListError);
