import React, {memo} from 'react';

import {IBaggageTDTariff} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {TVariantFareFamilyKeysByDirection} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';
import {EAviaClassType} from 'types/avia/EAviaClassType';

import {INormalizedTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';
import {IAviaContext} from 'reducers/avia/data-types';

import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {useMobile} from 'utilities/hooks/useMobile';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Box from 'components/Box/Box';
import Intersperse from 'components/Intersperse/Intersperse';
import Flight from 'projects/avia/pages/AviaOrder/components/Flight/Flight';
import AviaOrderTransfer from 'projects/avia/pages/AviaOrder/components/Transfer';
import OrderRouteTitle from 'projects/avia/pages/AviaOrder/components/OrderRouteTitle/OrderRouteTitle';

interface IAviaOrderRouteProps extends IWithQaAttributes {
    baggage: IBaggageTDTariff[];
    flights: IResultAviaFlight[];
    variant: IResultAviaVariant;
    selectedVariant: IResultAviaVariant;
    reference: INormalizedTDReference;
    hasBaggage: boolean;
    context: IAviaContext;
    fareFamilyKeys: TVariantFareFamilyKeysByDirection;
    isLastItem?: boolean;
}

const AviaOrderRoute: React.FC<IAviaOrderRouteProps> = props => {
    const {
        flights,
        variant,
        baggage,
        reference,
        hasBaggage,
        context,
        fareFamilyKeys,
        selectedVariant,
        isLastItem,
    } = props;
    const isMobile = useMobile();

    if (!flights.length) {
        return null;
    }

    const isBusiness = context.klass === EAviaClassType.BUSINESS;

    return (
        <Box below={isLastItem ? 0 : 8} between={isMobile ? 3 : 5}>
            <OrderRouteTitle
                flights={flights}
                variant={variant}
                fromTitle={flights[0].from.settlement?.title}
                toTitle={flights[flights.length - 1].to.settlement?.title}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'title',
                })}
            />

            <div>
                <Intersperse
                    separator={(idx): React.ReactNode => (
                        <AviaOrderTransfer
                            flight={flights[idx + 1]}
                            previousFlight={flights[idx]}
                        />
                    )}
                >
                    {flights.map((flight, index) => (
                        <Flight
                            key={flight.key}
                            flight={flight}
                            baggage={baggage[index]}
                            hasBaggage={hasBaggage}
                            isBusiness={isBusiness}
                            previousFlight={flights[index - 1]}
                            reference={reference}
                            fareFamilyKey={fareFamilyKeys[index]}
                            selectedVariant={selectedVariant}
                            {...prepareQaAttributes({
                                key: flight.key,
                                parent: props,
                                current: 'flight',
                            })}
                        />
                    ))}
                </Intersperse>
            </div>
        </Box>
    );
};

export default memo(AviaOrderRoute);
