import React from 'react';
import moment from 'moment';

import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {getTime} from 'projects/avia/lib/date/getTime';
import {getDuration} from 'projects/avia/lib/date/getDuration';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {HUMAN_DATE_WITH_FULL_WEEKDAY} from 'utilities/dateUtils/formats';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nTrainsTimeBlock from 'i18n/trains-time';
import * as i18nBlock from 'i18n/avia-Duration';

import Heading from 'components/Heading/Heading';
import DotSeparator from 'components/DotSeparator/DotSeparator';
import SelfConnectHint from 'projects/avia/pages/AviaOrder/components/SelfConnectHint/SelfConnectHint';

import cx from './OrderRouteTitle.scss';

interface IAviaOrderRouteTitleProps extends IWithQaAttributes {
    flights: IResultAviaFlight[];
    variant: IResultAviaVariant;
    className?: string;
    fromTitle: string | undefined;
    toTitle: string | undefined;
}

const AviaOrderRouteTitle: React.FC<IAviaOrderRouteTitleProps> = props => {
    const {fromTitle, toTitle, flights, variant, className} = props;

    const deviceType = useDeviceType();
    const {isMobile} = deviceType;

    const {departure} = flights[0];
    const {arrival} = flights[flights.length - 1];

    return (
        <div className={cx('root', className)}>
            <Heading
                className={cx('title')}
                level="3"
                size={isMobile ? 'l' : 'xl'}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'points',
                })}
            >
                {fromTitle} — {toTitle}
            </Heading>

            <div className={cx('subtitle')}>
                <span
                    className={cx('date')}
                    {...prepareQaAttributes({parent: props, current: 'dates'})}
                >
                    {moment(departure.local).format(
                        HUMAN_DATE_WITH_FULL_WEEKDAY,
                    )}
                </span>

                <DotSeparator />

                <span className={cx('duration')}>
                    {`${i18nBlock
                        .dDashHDashMDashFormat(
                            getDuration(getTime(arrival) - getTime(departure)),
                        )
                        .trim()} ${i18nTrainsTimeBlock.onDashWay()}`}
                </span>

                <DotSeparator />

                {flights.length > 1 && (
                    <SelfConnectHint
                        className={cx('selfConnectHint')}
                        variant={variant}
                    />
                )}
            </div>
        </div>
    );
};

export default React.memo(AviaOrderRouteTitle);
