import React, {useCallback, useEffect} from 'react';
import {useSelector} from 'react-redux';
import moment from 'moment';
import {noop} from 'lodash';

import {MINUTE} from 'utilities/dateUtils/constants';
import {DEFAULT_LANGUAGE_RU} from 'constants/common';

import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {ESubscriptionCode} from 'types/subscription/ESubscriptionCode';
import {ESubscriptionSource} from 'types/subscription/ESubscriptionSource';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';

import {getUserEmail} from 'selectors/common/userInfoSelector';

import {useAsync} from 'utilities/hooks/useAsync';
import {reachGoal} from 'utilities/metrika';
import wasClosedSaver from 'projects/avia/pages/AviaOrder/components/OrderSubscriptionModal/utilities/wasClosedSaver';
import {useIsUserSubscribed} from 'utilities/hooks/subscription/useIsUserSubscribed';

import SubscriptionModal from 'components/SubscriptionModal/SubscriptionModal';
import useUpdateNotification from 'components/UpdateNotification/hooks/useUpdateNotification';

import {notifierService} from 'serviceProvider';

const OrderSubscriptionModal: React.FC = () => {
    const {close: onClose, isVisible} = useUpdateNotification({
        timeout: 15 * MINUTE,
        onShowNotification() {
            if (wasClosedSaver.wasClosed) {
                return;
            }

            reachGoal(EAviaGoal.ORDER_SUBSCRIPTION_MODAL_SHOWED);
        },
    });

    const userEmail = useSelector(getUserEmail);
    const isUserSubscribed = useIsUserSubscribed();
    const [, subscribe] = useAsync(
        notifierService.provider()?.subscribe || noop,
    );

    useEffect(() => {
        if (!isVisible || wasClosedSaver.wasClosed) {
            return;
        }
    }, [isVisible, userEmail]);

    const handleSubscribe = useCallback(
        async (email: string) => {
            reachGoal(EAviaGoal.ORDER_SUBSCRIPTION_MODAL_SUBSCRIBED);
            subscribe({
                email,
                source: ESubscriptionSource.ORDER_MODAL,
                travelVerticalName: ESubscriptionVerticalName.Avia,
                promoSubscriptionCode: ESubscriptionCode.travelNews,
                timezone: moment.tz.guess(),
                language: DEFAULT_LANGUAGE_RU,
                nationalVersion: DEFAULT_LANGUAGE_RU,
            });
        },
        [subscribe],
    );

    const handleClose = useCallback(() => {
        wasClosedSaver.saveWasClosed();

        onClose();
    }, [onClose]);

    if (wasClosedSaver.wasClosed || (userEmail && isUserSubscribed)) {
        return null;
    }

    return (
        <SubscriptionModal
            email={userEmail}
            isVisible={isVisible}
            onClose={handleClose}
            onSubmit={handleSubscribe}
        />
    );
};

export default OrderSubscriptionModal;
