import React, {ReactNode} from 'react';

import {ECardWithDeviceLayoutVariation} from 'components/CardWithDeviceLayout/types/ECardWithDeviceLayoutVariation';

import {IAviaContext} from 'reducers/avia/data-types';

import {IAviaOrderOffers} from 'selectors/avia/order/aviaOrderSelector';
import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/avia-order';

import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import AviaOrderDisclaimers from 'projects/avia/pages/AviaOrder/components/Disclaimers/AviaOrderDisclaimers';
import AviaOfferListError from 'projects/avia/pages/AviaOrder/components/OfferListError/OfferListError';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import PriceListTitle from 'projects/avia/pages/AviaOrder/components/PriceListTitle/PriceListTitle';
import TariffSelector from 'projects/avia/components/AviaResultVariant/components/TariffSelector/TariffSelector';
import useUniqVariants from 'projects/avia/components/AviaResultVariant/components/TariffSelector/hooks/useUniqVariants';
import MobileTariffSelector from './components/MobileTariffSelector';
import AviaPriceList from 'projects/avia/components/AviaPriceList/AviaPriceList';

import ScopeContext from 'contexts/ScopeContext';

import cx from './PriceListBlock.scss';

interface IAviaOrderPriceListBlock extends IWithQaAttributes {
    context: IAviaContext;
    hasGoodPrice: boolean;
    error: boolean;
    totalSeats: number;
    baggagePrice: IPrice | null;
    partnersError: boolean;
    qid?: string;
    offers: IAviaOrderOffers | null;
    plusInfoBlock: React.ReactNode;
    selectedVariant: IResultAviaVariant | null;
    onSelectVariant(variant: IResultAviaVariant): void;
}

const PriceListBlock: React.FC<IAviaOrderPriceListBlock> = props => {
    const {
        context,
        error,
        hasGoodPrice,
        totalSeats,
        baggagePrice,
        partnersError,
        qid,
        offers,
        plusInfoBlock,
        selectedVariant,
        onSelectVariant,
    } = props;

    const deviceType = useDeviceType();
    const {isMobile} = deviceType;

    const variants = useUniqVariants(
        offers?.variantsByTariff.withoutBaggage.noOrUnknownRefund,
        offers?.variantsByTariff.withoutBaggage.chargedRefund,
        offers?.variantsByTariff.withoutBaggage.freeRefund,
        offers?.variantsByTariff.withBaggage.noOrUnknownRefund,
        offers?.variantsByTariff.withBaggage.chargedRefund,
        offers?.variantsByTariff.withBaggage.freeRefund,
    );

    if (error) {
        return <AviaOfferListError />;
    }

    const priceListTitle = (
        <PriceListTitle
            className={cx('titleItem')}
            totalSeats={totalSeats}
            klass={context.klass}
            {...prepareQaAttributes({
                parent: props,
                current: 'title',
            })}
        />
    );

    if (isMobile) {
        return (
            <>
                <CardWithDeviceLayout
                    variation={ECardWithDeviceLayoutVariation.ASIDE}
                    below="3"
                    shadow="none"
                    className={cx('root', {
                        priceList__hasGoodPrice: hasGoodPrice,
                    })}
                >
                    <Flex
                        className={cx('title')}
                        below="3"
                        between="4"
                        justifyContent="space-between"
                    >
                        {priceListTitle}

                        {selectedVariant && variants.length && (
                            <MobileTariffSelector
                                metricsContext="variant"
                                selectedVariant={selectedVariant}
                                variants={variants}
                                onSelect={onSelectVariant}
                                {...prepareQaAttributes(props)}
                            />
                        )}
                    </Flex>

                    {plusInfoBlock}

                    {partnersError && (
                        <Box className={cx('partnerError')}>
                            <Text size="s" color="alert">
                                {i18nBlock.partnerDashError()}
                            </Text>
                        </Box>
                    )}

                    <AviaPriceList
                        className={cx('offersList')}
                        qid={qid}
                        offerType={offers?.offerType}
                        cheapest={offers?.cheapest}
                        avia={offers?.avia}
                        other={offers?.other}
                        {...prepareQaAttributes(props)}
                    />
                </CardWithDeviceLayout>

                <AviaOrderDisclaimers />
            </>
        );
    }

    return (
        <>
            <Box below="5" between="2">
                {priceListTitle}
            </Box>

            <CardWithDeviceLayout
                variation={ECardWithDeviceLayoutVariation.ASIDE}
                below="5"
                className={cx('root', {
                    priceList__hasGoodPrice: hasGoodPrice,
                    priceList__cardWithBaggageButton: baggagePrice,
                })}
            >
                <Flex flexDirection="column" between={4}>
                    {selectedVariant && variants.length > 1 && (
                        <ScopeContext.Consumer>
                            {(scope): ReactNode => (
                                <TariffSelector
                                    scope={scope}
                                    metricsContext="variant"
                                    selectedVariant={selectedVariant}
                                    variants={variants}
                                    needToHideIfNoChoice
                                    onSelect={onSelectVariant}
                                    {...prepareQaAttributes(props)}
                                />
                            )}
                        </ScopeContext.Consumer>
                    )}

                    {plusInfoBlock}

                    {partnersError && (
                        <Box className={cx('partnerError')}>
                            <Text size="s" color="alert">
                                {i18nBlock.partnerDashError()}
                            </Text>
                        </Box>
                    )}

                    {offers && (
                        <AviaPriceList
                            qid={qid}
                            offerType={offers.offerType}
                            cheapest={offers.cheapest}
                            avia={offers.avia}
                            other={offers.other}
                            {...prepareQaAttributes(props)}
                        />
                    )}
                </Flex>
            </CardWithDeviceLayout>

            <AviaOrderDisclaimers />
        </>
    );
};

export default PriceListBlock;
