import React, {FC, useRef, useCallback} from 'react';

import {EAviaGoal} from 'utilities/metrika/types/goals/avia';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {
    reachChooseTariff,
    reachClickTariff,
    TTariffGoalContext,
} from 'projects/avia/utilities/tariffGoal';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useReachGoalThrottle} from 'utilities/metrika/useReachGoal';

import Tariffs from 'projects/avia/components/AviaResultVariant/components/TariffSelector/components/Tariffs/Tariffs';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';

import cx from './MobileTariffSelector.scss';

interface IMobileTariffSelectorProps extends IWithQaAttributes {
    metricsContext: TTariffGoalContext;
    selectedVariant: IResultAviaVariant;
    variants: IResultAviaVariant[];
    onSelect(variant: IResultAviaVariant): void;
}

const MobileTariffSelector: FC<IMobileTariffSelectorProps> = props => {
    const {selectedVariant, variants, metricsContext, onSelect} = props;

    const activeItemRef = useRef(null);

    const handleSelect = useCallback(
        (variantToSelect: IResultAviaVariant) => {
            reachChooseTariff(variantToSelect, metricsContext);
            reachClickTariff(variants, metricsContext);

            onSelect(variantToSelect);
        },
        [metricsContext, variants, onSelect],
    );

    const onTariffScroll = useReachGoalThrottle(EAviaGoal.TARIFF_SCROLL);

    if (variants.length === 1) {
        return null;
    }

    return (
        <HorizontalScroller
            padderClassName={cx('scrollPadder')}
            offset={4}
            initialScrollToRef={activeItemRef}
            onScroll={onTariffScroll}
            {...prepareQaAttributes(props)}
        >
            <Tariffs
                selectedVariant={selectedVariant}
                variants={variants}
                activeItemRef={activeItemRef}
                onSelect={handleSelect}
                showPriceHeader
                {...prepareQaAttributes({parent: props, current: 'tariffs'})}
            />
        </HorizontalScroller>
    );
};

export default React.memo(MobileTariffSelector);
