import React, {FunctionComponent, useRef} from 'react';

import {IWithClassName} from 'types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useTooltipHover from 'utilities/hooks/useTooltipHover';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nAviaOrderBlock from 'i18n/avia-AviaOrder';

import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import LinkButton from 'components/LinkButton/LinkButton';
import InfoIcon from 'icons/16/Info';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';
import BottomSheet from 'components/BottomSheet/BottomSheet';
import Box from 'components/Box/Box';
import SelfConnectLSVG from 'icons/images/SelfConnectL.svg';
import ImageIcon from 'components/ImageIcon/ImageIcon';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';

import cx from './SelfConnectHint.scss';

export interface ISelfSelfConnectHintProps extends IWithClassName {
    variant: IResultAviaVariant;
}

interface ISelfConnectHintDeviceProps extends IWithClassName {
    trigger: React.ReactElement;
    description: React.ReactElement;
}

const SelfConnectHintMobile: FunctionComponent<ISelfConnectHintDeviceProps> = ({
    trigger,
    description,
    className,
}) => {
    const {value: isVisible, setTrue: show, setFalse: hide} = useBoolean(false);
    const triggerRef = useRef(null);

    return (
        <>
            <LinkButton
                theme="alert"
                className={cx(className)}
                onClick={show}
                innerRef={triggerRef}
            >
                {trigger}
            </LinkButton>

            <BottomSheet isOpened={isVisible} onClose={hide}>
                {description}
            </BottomSheet>
        </>
    );
};
const DIRECTION_DEFAULT = [EPopupDirection.BOTTOM, EPopupDirection.TOP];

const SelfConnectHintDesktop: FunctionComponent<ISelfConnectHintDeviceProps> =
    ({trigger, description, className}) => {
        const {isHovered, handleMouseEnter, handleMouseLeave} =
            useTooltipHover();
        const triggerRef = useRef(null);

        return (
            <>
                <LinkButton
                    theme="alert"
                    className={cx(className)}
                    onMouseEnter={handleMouseEnter}
                    onMouseLeave={handleMouseLeave}
                    innerRef={triggerRef}
                >
                    {trigger}
                </LinkButton>

                <MessageBoxPopup
                    isVisible={isHovered}
                    anchorRef={triggerRef}
                    direction={DIRECTION_DEFAULT}
                    onClose={handleMouseLeave}
                >
                    {description}
                </MessageBoxPopup>
            </>
        );
    };

const SelfConnectHint: FunctionComponent<ISelfSelfConnectHintProps> = ({
    variant: {
        price: {selfConnect},
    },
    className,
}) => {
    const deviceType = useDeviceType();
    const {isMobile, isDesktop} = deviceType;

    const trigger = (
        <TextWithIcon
            className={cx('selfConnect')}
            size="m"
            spaceBetween={1}
            text={i18nAviaOrderBlock.selfConnectShortMessage()}
            iconRight={InfoIcon}
        />
    );

    const description = (
        <Box
            className={cx('description', deviceMods('description', deviceType))}
            between="5"
        >
            {isMobile && (
                <div className={cx('illustration')}>
                    <ImageIcon src={SelfConnectLSVG} width={220} height={220} />
                </div>
            )}
            <Heading className={cx('heading')} level={isMobile ? 2 : 3}>
                {i18nAviaOrderBlock.selfConnectShortMessage()}
            </Heading>
            <Text
                className={cx('text')}
                size="s"
                color={isMobile ? 'primary' : 'inverse'}
            >
                {i18nAviaOrderBlock.selfConnectLongMessage()}
            </Text>
        </Box>
    );

    return selfConnect ? (
        <>
            {isDesktop && (
                <SelfConnectHintDesktop
                    trigger={trigger}
                    description={description}
                    className={className}
                />
            )}
            {isMobile && (
                <SelfConnectHintMobile
                    trigger={trigger}
                    description={description}
                    className={className}
                />
            )}
        </>
    ) : null;
};

export default SelfConnectHint;
