import {
    FC,
    RefObject,
    useCallback,
    useEffect,
    useMemo,
    useRef,
    useState,
} from 'react';
import _isEqual from 'lodash/isEqual';
import _throttle from 'lodash/throttle';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';

import {useEventListener} from 'utilities/hooks/useEventListener';
import {useToggle} from 'utilities/hooks/useToggle';

import * as i18nBlock from 'i18n/avia-order';

import Card from 'components/Card/Card';
import Tariffs from 'projects/avia/components/AviaResultVariant/components/TariffSelector/components/Tariffs/Tariffs';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import Text from 'components/Text/Text';
import Box from 'components/Box/Box';
import TariffsPreview from 'projects/avia/pages/AviaOrder/components/TariffsBottomSheet/components/TariffsPreview/TariffsPreview';
import BottomSheet from 'components/BottomSheet/BottomSheet';

import {useOnBuyHandler} from 'projects/avia/hooks/useOnBuyHandler';
import {useAviaOrderContext} from 'projects/avia/pages/AviaOrder/context/AviaOrderContext';

import cx from './TariffsBottomSheet.scss';

const THROTTLE_WAIT = 100;

export interface IFixedMobileTariffsSelector {
    variants: IResultAviaVariant[];
    offers: IResultAviaVariant[];
    selectedVariant: IResultAviaVariant | null;
    onSelect?(variant: IResultAviaVariant): void;
}

const TariffsBottomSheet: FC<IFixedMobileTariffsSelector> = ({
    variants,
    offers,
    selectedVariant,
    onSelect,
}) => {
    const [isBottomSheetOpen, toggleBottomSheetOpen] = useToggle(false);
    const [scrollLeft, setScrollLeft] = useState(0);
    const [isFixedPreviewVisible, setIsFixedPreviewVisible] = useState(false);
    const tariffsScrollerRef = useRef<HTMLDivElement>(null);
    const previewScrollerRef = useRef<HTMLDivElement>(null);
    const bottomSheetScrollerRef = useRef<HTMLDivElement>(null);
    const tariffSelectorRef = useRef<HTMLDivElement>(null);
    const fixedPreviewRef = useRef<HTMLDivElement>(null);
    const tariffsScrollerActiveFixedItemRef = useRef(null);

    const {modalRootRef} = useAviaOrderContext();

    const onBuyClick = useOnBuyHandler();

    const handleCalculateFixedPreviewVisibility = useCallback(() => {
        const posTop =
            tariffSelectorRef.current?.getBoundingClientRect().top || 0;
        const fixedPreviewPos =
            window.innerHeight - (fixedPreviewRef.current?.clientHeight || 0);

        setIsFixedPreviewVisible(posTop > fixedPreviewPos);
    }, []);

    useEventListener(
        'scroll',
        _throttle(handleCalculateFixedPreviewVisibility, THROTTLE_WAIT),
        modalRootRef || undefined,
    );

    useEffect(handleCalculateFixedPreviewVisibility, []);

    useEffect(() => {
        setTimeout(() => {
            bottomSheetScrollerRef.current?.scrollTo(scrollLeft, 0);
        }, 100);
    }, [isBottomSheetOpen]);

    const onTariffScrollThrottle = useCallback(
        _throttle((ref: RefObject<HTMLDivElement>) => {
            const scrollLeftTo = ref.current?.scrollLeft || 0;

            setScrollLeft(scrollLeftTo);

            if (!_isEqual(previewScrollerRef.current, ref.current)) {
                previewScrollerRef.current?.scrollTo(scrollLeftTo, 0);
            }

            if (!_isEqual(tariffsScrollerRef.current, ref.current)) {
                tariffsScrollerRef.current?.scrollTo(scrollLeftTo, 0);
            }
        }, THROTTLE_WAIT),
        [],
    );

    const onTariffScroll = useCallback(
        (ref: RefObject<HTMLDivElement>) => () => {
            onTariffScrollThrottle(ref);
        },
        [onTariffScrollThrottle],
    );

    const title = useMemo(
        () => (
            <Text size="l" weight="bold">
                {i18nBlock.chooseTariff()}
            </Text>
        ),
        [],
    );

    return (
        <>
            <Card shadow="none" x={4} y={4} below={3} ref={tariffSelectorRef}>
                {title}
                <HorizontalScroller
                    className={cx('scroller')}
                    padderClassName={cx('scrollPadder')}
                    offset={4}
                    stopPropagationTouchmove
                    scrollerRef={tariffsScrollerRef}
                    onScroll={onTariffScroll(tariffsScrollerRef)}
                    initialScrollToRef={tariffsScrollerActiveFixedItemRef}
                >
                    <Tariffs
                        variants={variants}
                        offers={offers}
                        onPriceButtonClick={onBuyClick}
                        selectedVariant={selectedVariant}
                        onSelect={onSelect}
                        activeItemRef={tariffsScrollerActiveFixedItemRef}
                        showPriceButton
                    />
                </HorizontalScroller>
            </Card>

            <div
                className={cx('fixedPreview', {
                    fixedPreview_hidden: !isFixedPreviewVisible,
                })}
                onClick={toggleBottomSheetOpen}
                ref={fixedPreviewRef}
            >
                <Box x={4} y={4}>
                    {title}
                    <HorizontalScroller
                        offset={4}
                        stopPropagationTouchmove
                        className={cx('scroller')}
                        scrollerRef={previewScrollerRef}
                        onScroll={onTariffScroll(previewScrollerRef)}
                    >
                        <TariffsPreview
                            selectedVariant={selectedVariant}
                            variants={variants}
                            onSelect={onSelect}
                        />
                    </HorizontalScroller>
                </Box>
            </div>

            <BottomSheet
                isOpened={isBottomSheetOpen}
                onClose={toggleBottomSheetOpen}
            >
                {title}
                <HorizontalScroller
                    className={cx('scroller')}
                    padderClassName={cx('scrollPadder')}
                    offset={4}
                    stopPropagationTouchmove
                    scrollerRef={bottomSheetScrollerRef}
                    onScroll={onTariffScroll(bottomSheetScrollerRef)}
                >
                    <Tariffs
                        variants={variants}
                        offers={offers}
                        onPriceButtonClick={onBuyClick}
                        selectedVariant={selectedVariant}
                        onSelect={onSelect}
                        showPriceButton
                    />
                </HorizontalScroller>
            </BottomSheet>
        </>
    );
};

export default TariffsBottomSheet;
