import moment from 'moment';

import {IResultAviaFlight} from 'selectors/avia/utils/denormalization/flight';

import {CHAR_MIDDLE_DASH} from 'utilities/strings/charCodes';
import {getTime} from 'projects/avia/lib/date/getTime';
import {getDuration} from 'projects/avia/lib/date/getDuration';

import * as i18nBlock from 'i18n/avia-Duration';
import * as i18nSearchBlock from 'i18n/avia-search';

import ConnectionIcon from 'icons/12/Connection';
import NightConnectionIcon from 'icons/12/NightConnection';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';

import cx from './styles.scss';

interface IAviaOrderTransferProps {
    flight: IResultAviaFlight;
    previousFlight: IResultAviaFlight;
}

export default function AviaOrderTransfer({
    flight,
    previousFlight,
}: IAviaOrderTransferProps) {
    const {arrival, to} = previousFlight;
    const {departure, from} = flight;

    const airportChange = to.id !== from.id;
    const isNightTransfer = !moment(arrival.local).isSame(
        departure.local,
        'day',
    );

    const whereTransfer = i18nSearchBlock.transferInCityWithPreposition({
        isNightTransfer,
        preposition: to.settlement?.preposition ?? '',
        title: to.settlement?.phraseIn ?? '',
    });
    const timeTransfer = i18nBlock
        .dDashHDashMDashFormat(
            getDuration(getTime(departure) - getTime(arrival)),
        )
        .trim();

    return (
        <div className={cx('root')}>
            <div
                className={cx(
                    'transferBlock',
                    isNightTransfer && 'transferBlock_nightTransfer',
                )}
            >
                <TextWithIcon
                    size="m"
                    text={`${whereTransfer} ${CHAR_MIDDLE_DASH} ${timeTransfer}`}
                    iconLeft={
                        isNightTransfer ? NightConnectionIcon : ConnectionIcon
                    }
                    iconSize={12}
                />
            </div>

            {airportChange && (
                <div
                    className={cx(
                        'transferBlock',
                        'transferBlock_airportChange',
                    )}
                >
                    <TextWithIcon
                        size="m"
                        text={i18nSearchBlock.transferDotAirportDashChange()}
                        iconLeft={ConnectionIcon}
                    />
                </div>
            )}
        </div>
    );
}
