import {useCallback, useMemo, useState} from 'react';
import first from 'lodash/first';

import {
    AVIA_BAGGAGE_PARAM_KEY,
    AVIA_CHARGE_REFUND_PARAM_KEY,
    AVIA_FREE_REFUND_PARAM_KEY,
} from 'projects/avia/constants/queryParams';

import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';

import {IResultAviaVariant} from 'selectors/avia/utils/denormalization/variant';
import {IVariantsByTariff} from 'selectors/avia/order/aviaOrderSelector';

import updateLocationWithQuery from 'utilities/updateLocationWithQuery/updateLocationWithQuery';
import useQuery from 'utilities/hooks/useQuery';

export default function useVariants(
    variantsByTariff: IVariantsByTariff | undefined,
): {
    variants: IResultAviaVariant[] | null;
    selectedVariant: IResultAviaVariant | null;
    handleSelect(variantToSelect: IResultAviaVariant): void;
} {
    const aviaParams = useQuery() as IAviaParams;

    const [hasBaggage, setHasBaggage] = useState(
        aviaParams[AVIA_BAGGAGE_PARAM_KEY] === '1',
    );
    const [hasFreeRefund, setHasFreeRefund] = useState(
        aviaParams[AVIA_FREE_REFUND_PARAM_KEY] === '1',
    );
    const [hasChargedRefund, setHasChargedRefund] = useState(
        aviaParams[AVIA_CHARGE_REFUND_PARAM_KEY] === '1',
    );

    const variants = useMemo(() => {
        if (!variantsByTariff) {
            return null;
        }

        const result = hasBaggage
            ? variantsByTariff.withBaggage
            : variantsByTariff.withoutBaggage;

        if (hasFreeRefund) {
            return result.freeRefund;
        }

        if (hasChargedRefund) {
            return result.chargedRefund;
        }

        return result.noOrUnknownRefund;
    }, [hasBaggage, hasChargedRefund, hasFreeRefund, variantsByTariff]);

    const handleSelectVariant = useCallback(
        (variantToSelect: IResultAviaVariant) => {
            const variantHasBaggage = variantToSelect.hasBaggage;
            const variantHasFreeRefund = variantToSelect.price.hasFreeRefund;
            const variantHasChargeRefund =
                variantToSelect.price.hasChargedRefund;

            setHasBaggage(variantHasBaggage);
            setHasFreeRefund(variantHasFreeRefund);
            setHasChargedRefund(variantHasChargeRefund);

            updateLocationWithQuery(
                {
                    [AVIA_BAGGAGE_PARAM_KEY]: variantHasBaggage
                        ? '1'
                        : undefined,
                    [AVIA_FREE_REFUND_PARAM_KEY]: variantHasFreeRefund
                        ? '1'
                        : undefined,
                    [AVIA_CHARGE_REFUND_PARAM_KEY]: variantHasChargeRefund
                        ? '1'
                        : undefined,
                },
                undefined,
                {replace: true, needSaveLocationState: true},
            );
        },
        [],
    );

    const selectedVariant = useMemo(() => first(variants) ?? null, [variants]);

    return {
        variants,
        selectedVariant,
        handleSelect: handleSelectVariant,
    };
}
