import React, {useEffect, useState} from 'react';

import {useMobile} from 'utilities/hooks/useMobile';
import {CHAR_EM_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';
import {formatPeriod} from 'utilities/dateUtils/formatPeriod';

import * as i18nBlock from 'i18n/avia-AviaRedirect-AviaRedirectProgress';

import Card from 'components/Card/Card';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import FlightIcon from 'icons/16/Flight';
import AviaPartnerLogo from 'projects/avia/components/PartnerLogo/PartnerLogo';

import {IPartnerInfo} from 'server/api/AviaPartnersApi/AviaPartnersApi';

import cx from './AviaRedirectProgress.scss';

interface IAviaRedirectProgressProps {
    partner: IPartnerInfo | null;
    theme?: 'aero';
    title?: string;
    fromCity: string;
    toCity: string;
    dateForward: string;
    dateBackward?: string;
    boy: boolean;
}

const AviaRedirectProgress: React.FC<IAviaRedirectProgressProps> = ({
    partner,
    fromCity,
    toCity,
    theme,
    title = i18nBlock.redirectDashProgressDashTitle(),
    dateForward,
    dateBackward,
    boy,
}) => {
    const isMobile = useMobile();
    const [start, setStart] = useState(false);

    useEffect(() => setStart(true), []);

    const direction = `${fromCity}${CHAR_NBSP}${CHAR_EM_DASH} ${toCity}`;
    const dates = formatPeriod(dateForward, dateBackward);

    return (
        <div className={cx('overflow-buble')}>
            <Card
                shadow="default"
                className={cx('root', theme && `root_${theme}`, {
                    root_mobile: isMobile,
                    root_start: start,
                })}
            >
                <div className={cx('progressbar-wrapper')}>
                    <div className={cx('progressbar')} />
                    <div className={cx('progressbar-icon-wrapper')}>
                        <div className={cx('progressbar-icon-pusher')} />
                        <div className={cx('progressbar-icon')}>
                            <FlightIcon />
                        </div>
                    </div>
                </div>

                <div className={cx('info')}>
                    <Heading className={cx('title')} level={isMobile ? 2 : 1}>
                        {title}
                    </Heading>

                    <AviaPartnerLogo
                        partner={partner}
                        size={isMobile ? 's' : 'm'}
                        className={cx('partner-logo', {
                            'partner-logo__boy': boy,
                        })}
                    />

                    <Text
                        className={cx('redirect-text')}
                        size={isMobile ? 'm' : 'l'}
                    >
                        {i18nBlock.redirectDashProgressDashSubtitle()}
                    </Text>

                    <Text
                        className={cx('redirect-text')}
                        size={isMobile ? 'm' : 'l'}
                    >
                        {`${direction}, ${dates}`}
                    </Text>
                </div>
            </Card>
        </div>
    );
};

export default AviaRedirectProgress;
