import React, {useEffect, useMemo} from 'react';
import {useSelector} from 'react-redux';

import {URLs} from 'constants/urls';
import {EAppActions} from 'constants/platforms/TPlatforms';

import {EAviaGoal} from 'utilities/metrika/types/goals/avia';

import {aviaContextSelector} from 'selectors/avia/context/aviaContextSelector';
import {isFromXredirectSelector} from 'selectors/common/isFromXredirectSelector';

import {reachGoal} from 'utilities/metrika';
import {isValidAviaSearchParams} from 'projects/avia/lib/search/isValidAviaSearchParams';
import {getSearchFormFromContext} from 'projects/avia/lib/search/getSearchFormFromContext';
import {pickAviaSearchFormParams} from 'projects/avia/lib/search/pickAviaSearchFormParams';
import {aviaURLs} from 'projects/avia/lib/urls';

import * as i18nBlockAviaRedirect from 'i18n/avia-redirect';

import AviaLayout from 'projects/avia/components/AviaLayout';
import LayoutError500 from 'components/Layouts/LayoutError500/LayoutError500';
import DocumentMeta from 'components/DocumentMeta/DocumentMeta';
import {TErrorActionType} from 'components/ErrorModal/ErrorModal';

import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';
import {useCoordinator, usePlatform} from 'contexts/PlatformContext';

import {aviaRedirectErrorController} from 'server/redux/avia/redirectErrorController';

const AviaRedirectError: React.FC = () => {
    useEffect(() => {
        reachGoal(EAviaGoal.REDIRECT_SHOW_ERROR_PAGE);
    }, []);

    const context = useSelector(aviaContextSelector);
    const isFromXredirect = useSelector(isFromXredirectSelector);

    const {isWeb} = usePlatform();
    const coordinator = useCoordinator();

    const searchForm = getSearchFormFromContext(context);
    const filteredSearchForm = pickAviaSearchFormParams(searchForm);
    const actionUrl = isValidAviaSearchParams(filteredSearchForm)
        ? aviaURLs.getSearchResultsUrl(filteredSearchForm)
        : URLs.avia;
    const redirectButtonTitle =
        i18nBlockAviaRedirect.redirectDashErrorDashAction();

    const action = useMemo<TErrorActionType | undefined>(() => {
        if (isFromXredirect) {
            return undefined;
        }

        if (!isWeb) {
            const handler: React.MouseEventHandler = e => {
                const processed = coordinator.doAction(
                    EAppActions.NAVIGATE_TO_SEARCH_AVIA_PAGE,
                );

                if (processed) {
                    e.preventDefault();

                    return;
                }
            };

            return {
                type: 'button',
                title: redirectButtonTitle,
                handler,
            };
        }

        return {
            type: 'button-link',
            title: redirectButtonTitle,
            props: {
                to: actionUrl,
            },
        };
    }, [actionUrl, coordinator, isFromXredirect, isWeb, redirectButtonTitle]);

    return (
        <AviaLayout
            showHeader={isFromXredirect ? false : undefined}
            showFooter={isFromXredirect ? false : undefined}
            hideSearchInformation={!isWeb}
        >
            <DocumentMeta
                title={i18nBlockAviaRedirect.redirectDashErrorDashMetaDashTitle()}
                description={i18nBlockAviaRedirect.redirectDashErrorDashMetaDashDescription()}
            />

            <LayoutError500
                title={i18nBlockAviaRedirect.redirectDashErrorDashTitle()}
                subtitle={i18nBlockAviaRedirect.redirectDashErrorDashSubtitle()}
                action={action}
            />
        </AviaLayout>
    );
};

export default serverFetchDataDispatcher([aviaRedirectErrorController])(
    AviaRedirectError,
);
