import {useEffect, useMemo} from 'react';
import {useDispatch} from 'react-redux';
import {useLocation} from 'react-router-dom';
import {debounce} from 'lodash';

import {setFilterValues} from 'reducers/avia/search/results/filters/actions';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    hashToFilterValues,
    TPartialFilterValues,
} from 'projects/avia/lib/search/filters/converters';

const URL_UPDATE_TIMEOUT = 100;

function useFilterValueSync(): void {
    const {isMobile} = useDeviceType();
    const dispatch = useDispatch();
    const location = useLocation();
    const updateFilters = useMemo(() => {
        const func = (filterValues: TPartialFilterValues): void => {
            dispatch(setFilterValues(filterValues));
        };

        // На мобильных устройствах фильтры открываются в модале с использованием useSupportHistoryBack
        // Проблема:
        //
        // X - урл до применения фильтров
        // Y - фейковый урл модала
        // Z - новый урл после применения фильтров
        //
        // - открываем фильтры, history.push - X -> Y
        // - применяем фильтры, history.replace - Y -> Z
        // - модал закрывается, history.back - Z -> X (срабатывает этот хук и мы рисуем старую выдачу)
        // - модал запомнил, что пока он был открыт произошел replace
        //   и применяет его к старому урлу - X -> Z (тут сноова срабатывает хук и мы рисуем отфильтрованную выдачу)
        //
        // Чтобы избежать обновлений на момент восстановления урла добавлен debounce
        return isMobile ? debounce(func, URL_UPDATE_TIMEOUT) : func;
    }, [isMobile, dispatch]);

    useEffect(() => {
        const hash = location.hash.slice(1);
        const filterValues = hashToFilterValues(hash);

        updateFilters(filterValues);
    }, [updateFilters, location]);
}

// Почему в виде компонента: внутри useFilterValueSync используется хук
// useHistory, который срабатывает на изменение урла и может приводить к нежелательным апдейтам
export default function AviaFilterValueSync() {
    useFilterValueSync();

    return null;
}
