import React, {RefObject, useCallback, useEffect, useMemo, useRef} from 'react';
import {RouteComponentProps} from 'react-router-dom';
import {parse, ParsedQuery} from 'query-string';
import {useDispatch, useSelector} from 'react-redux';

import {
    DEFAULT_ADULTS_COUNT,
    DEFAULT_CHILDREN_COUNT,
    DEFAULT_INFANTS_COUNT,
} from 'constants/avia';

import {EAviaActionLogPageName} from 'server/loggers/avia/AviaActionLog/types/EAviaActionLogPageName';
import {EAviaUserSearchLogPages} from 'server/loggers/avia/AviaUserSearchLog/types/AVIA_USER_SEARCH_LOG_PAGES';
import {EAviaClassType} from 'types/avia/EAviaClassType';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {EAviaInitSearchError} from 'types/avia/search/EAviaInitSearchError';

import {withPrevState} from 'reducers/utils/withPrevState';
import {initSearch} from 'reducers/avia/search/results/actions';
import {EAviaResultsSortType} from 'reducers/avia/search/results/EAviaResultsSortType';
import {setAviaCurrentPage} from 'reducers/avia/page/actions';
import {aviaOrderActions} from 'reducers/avia/order/actions';

import {aviaSearchResultState} from 'selectors/avia/search/aviaSearchResultState';
import {aviaContextSelector} from 'selectors/avia/context/aviaContextSelector';
import {lowerCrossSaleEnabledSelector} from 'selectors/avia/search/lowerCrossSaleEnabledSelector';
import {aviaCleanSerpEnabledSelector} from 'selectors/avia/search/aviaCleanSerpEnabledSelector';
import {getAviaCountryRestrictions} from 'selectors/avia/aviaSelectors';
import {getWeatherForecastStateSelector} from 'selectors/avia/weatherForecast/getWeatherForecastStateSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {isAviaSearchParams} from 'projects/avia/lib/search/isAviaSearchParams';
import {
    getAviaSearchParamsKey,
    isSameAviaSearchParamsKeys,
} from 'projects/avia/lib/search/aviaSearchParamsKey';
import {reachGoal} from 'utilities/metrika';
import {EAviaRumEvents} from 'projects/avia/lib/EAviaRumEvents';
import {deviceMods} from 'utilities/stylesUtils';
import {useToggle} from 'utilities/hooks/useToggle';
import {parseAviaPeriod} from 'projects/avia/lib/search/parsePeriod';
import {getAviaRedirectOptionsByQuery} from './utilities/getAviaRedirectOptionsByQuery';
import {useExperiments} from 'utilities/hooks/useExperiments';

import {useHeaderScrollSlide} from 'hooks/useHeaderScrollSlide';

import Box from 'components/Box/Box';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import Container from 'components/Layouts/Container/Container';
import RedirectWithStatus from 'components/RedirectWithStatus/RedirectWithStatus';
import {TwoColumnLayout} from 'components/Layouts/TwoColumnLayout/TwoColumnLayout';
import AviaSearchLayout from 'projects/avia/pages/AviaSearchResult/components/AviaSearchLayout/AviaSearchLayout';
import {AviaSearchProgressBar} from 'projects/avia/components/AviaSearchProgressBar/AviaSearchProgressBar';
import AviaSearchInvalidState from './components/AviaSearchInvalidState/AviaSearchInvalidState';
import {UpdateNotification} from 'projects/avia/components/UpdateNotification/UpdateNotification';
import AviaSearchDisclaimers from 'projects/avia/pages/AviaSearchResult/components/Disclaimers/Disclaimers';
import AviaResetFiltersBlock from 'projects/avia/pages/AviaSearchResult/components/AviaResetFiltersBlock';
import AdBlock from 'projects/avia/pages/AviaSearchResult/components/AdBlock/AdBlock';
import AviaSearchTopResult from './components/AviaSearchTopResult/AviaSearchTopResult';
import AviaSearchRestResults, {
    SLICE_SIZE,
} from './components/AviaSearchRestResults/AviaSearchRestResults';
import EmptySerpWithContainer from 'components/EmptySerpWithContainer/EmptySerpWithContainer';
import AviaSubscriptionAndDynamicAsButtons from 'projects/avia/pages/AviaSearchResult/components/AviaSubscriptionAndDynamicAsButtons/AviaSubscriptionAndDynamicAsButtons';
import DesktopDynamics from 'projects/avia/pages/AviaSearchResult/components/DesktopDynamics/DesktopDynamics';
import CountryRestrictions from 'components/CountryRestrictions/CountryRestrictions';
import DirectWithUpdater, {
    IDirectWithUpdaterRef,
} from './components/DirectWithUpdater/DirectWithUpdater';
import AviaSearchResultsSortingSelect, {
    TAviaSearchResultsSortingSelectValue,
} from 'projects/avia/components/AviaSearchResultsSortingSelect/AviaSearchResultsSortingSelect';
import AviaHotelsCrossSaleMapSidebar from 'projects/avia/components/AviaHotelsCrossSaleMapSidebar/AviaHotelsCrossSaleMapSidebar';

import {Order} from 'projects/avia/pages/AviaOrder/AviaOrderPage';
import CrossSaleMap from 'projects/avia/containers/CrossSaleMap/CrossSaleMap';
import useSearchMetrika from 'projects/avia/pages/AviaSearchResult/hooks/useSearchMetrika';
import {useAviaSortingFilter} from 'projects/avia/hooks/useAviaSortingFilter';
import AviaWeatherForecast from 'projects/avia/containers/AviaWeatherForecast/AviaWeatherForecast';

import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';
import {useRumObserveState, useRumUi} from 'contexts/RumUiContext';

import prefetchSearchData from 'server/redux/avia/pages/search/dataFetcher';

import {aviaLoggingBrowserProvider} from 'serviceProvider/avia/logging/aviaLoggingBrowserProvider';

import {aviaSearchResultPageSelector} from './aviaSearchResultPageSelector';
import {useInvalidAviaSearchLayoutProps} from './hooks/useInvalidAviaSearchLayoutProps';

import cx from './AviaSearchResult.scss';

type TResultPageSelectorData = Pick<
    ReturnType<typeof aviaSearchResultPageSelector>,
    'searchVariantsView' | 'klass'
> &
    Pick<
        ReturnType<typeof aviaSearchResultState>,
        'cont' | 'qid' | 'progress' | 'error'
    >;

interface IAviaSearchResultProps
    extends TResultPageSelectorData,
        IWithQaAttributes {
    isSortByInterest: boolean;
    noVisibleVariants: boolean;
    searchIsCompleted: boolean;
    isRoundTrip: boolean;
    directRef: RefObject<IDirectWithUpdaterRef>;
    isBackground: boolean;
    /** Название страны из пункта "откуда" на русском языке */
    fromCountryTitle: string | undefined;
    /** Название страны из пункта "куда" на русском языке */
    toCountryTitle: string | undefined;

    initSearchCallback: () => void;
    updateDirect: () => void;
    lowerCrossSaleEnabled: boolean;
}

const MIN_SKELETONS_TIME = 2000;
const ROOT_QA = 'avia-search';

const AviaSearchResultMemo: React.FC<IAviaSearchResultProps> = React.memo(
    // TODO здесь нужен рефакторинг компонента
    // eslint-disable-next-line complexity
    function AviaSearchResult({
        cont,
        searchVariantsView,
        qid,
        progress,
        searchIsCompleted,
        error,
        klass,
        directRef,
        fromCountryTitle,
        toCountryTitle,
        isSortByInterest,
        isRoundTrip,
        noVisibleVariants,
        updateDirect,
        lowerCrossSaleEnabled,
        isBackground,
    }): React.ReactElement {
        const isLoading = cont !== null;
        const isInitialVariantsLoading =
            !searchIsCompleted && !error && noVisibleVariants;
        const showSubscriptionAndDynamic = klass === EAviaClassType.ECONOMY;
        const deviceType = useDeviceType();
        const {
            aviaNewTouchFilters,
            aviaWeatherDesktop,
            aviaOnlyWeatherRight,
            aviaOnlyHotelsRight,
            aviaHotelsThenWeatherRight,
        } = useExperiments();

        const [dynamicsAreOpen, toggleDynamics] = useToggle(false);
        const invalidAviaSearchLayoutProps =
            useInvalidAviaSearchLayoutProps(error);
        const aviaCleanSerpEnabled = useSelector(aviaCleanSerpEnabledSelector);
        const countryRestrictions = useSelector(getAviaCountryRestrictions);
        const weatherForecast = useSelector(getWeatherForecastStateSelector);

        const {isVisible} = useHeaderScrollSlide();

        const {sorting, handleSortChange: changeSort} = useAviaSortingFilter();

        const isWithoutDirectBlock =
            (aviaWeatherDesktop ||
                aviaOnlyWeatherRight ||
                aviaOnlyHotelsRight ||
                aviaHotelsThenWeatherRight) &&
            deviceType.isDesktop;

        const handleSortChange = useCallback(
            ([type, direction]: TAviaSearchResultsSortingSelectValue) =>
                changeSort({type, direction}),
            [changeSort],
        );

        const crossSaleNode = useMemo(() => {
            if (deviceType.isDesktop && aviaWeatherDesktop) {
                return null;
            }

            if (aviaCleanSerpEnabled) {
                return null;
            }

            return (
                <CrossSaleMap
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'hotelsCrossSaleMap',
                    })}
                />
            );
        }, [aviaCleanSerpEnabled, aviaWeatherDesktop, deviceType.isDesktop]);

        if (error === EAviaInitSearchError.SAME_POINTS) {
            return (
                <AviaSearchLayout searchFormInitialIsExpanded>
                    <EmptySerpWithContainer
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'emptySerp',
                        })}
                    />
                </AviaSearchLayout>
            );
        }

        if (
            (error || (searchIsCompleted && noVisibleVariants)) &&
            !dynamicsAreOpen
        ) {
            return (
                <AviaSearchLayout {...invalidAviaSearchLayoutProps}>
                    <Container
                        className={cx(
                            'invalidStateBannerWrapper',
                            deviceMods('invalidStateBannerWrapper', deviceType),
                        )}
                    >
                        <AdFoxBanner
                            type={EAdFoxBannerType.Inline}
                            position={EAdFoxBannerPosition.Center}
                        />
                    </Container>
                    <AviaSearchInvalidState error={error} />
                </AviaSearchLayout>
            );
        }

        return (
            <AviaSearchLayout
                isSkeletons={isInitialVariantsLoading}
                filtersAreVisible
            >
                <Container
                    className={cx(
                        'resultsContainer',
                        deviceMods('resultsContainer', deviceType),
                        {
                            resultsContainer_withMarginTop:
                                deviceType.isMobile && aviaNewTouchFilters,
                        },
                    )}
                    provideScope
                >
                    <TwoColumnLayout
                        rightColumnOffset={10}
                        rightColumnWidth={75}
                        deviceType={deviceType}
                    >
                        <TwoColumnLayout.LeftColumn>
                            <div
                                {...prepareQaAttributes(
                                    `avia-search-is-${
                                        isLoading || isInitialVariantsLoading
                                            ? 'in-process'
                                            : 'completed'
                                    }`,
                                )}
                            >
                                {deviceType.isMobile && !aviaNewTouchFilters && (
                                    <Box above={2} below={5}>
                                        <AviaSearchResultsSortingSelect
                                            onChange={handleSortChange}
                                            value={[
                                                sorting.type,
                                                sorting.direction,
                                            ]}
                                            size="l"
                                            {...prepareQaAttributes(
                                                'avia-sorting',
                                            )}
                                        />
                                    </Box>
                                )}

                                {!aviaCleanSerpEnabled && (
                                    <AdFoxBanner
                                        className={cx(
                                            'banner',
                                            deviceMods('banner', deviceType),
                                        )}
                                        type={EAdFoxBannerType.Inline}
                                        position={EAdFoxBannerPosition.Center}
                                    />
                                )}

                                {fromCountryTitle && toCountryTitle && (
                                    <CountryRestrictions
                                        className={cx('countryRestrictions')}
                                        countryRestrictions={
                                            countryRestrictions
                                        }
                                        forceSkeletons={
                                            isInitialVariantsLoading
                                        }
                                        fromCountryTitle={fromCountryTitle}
                                        toCountryTitle={toCountryTitle}
                                    />
                                )}

                                <Box className={cx('serp')}>
                                    <AviaSearchTopResult
                                        variants={searchVariantsView.top}
                                        isSkeletons={isInitialVariantsLoading}
                                        isSortByInterest={isSortByInterest}
                                        isRoundTrip={isRoundTrip}
                                        minLoaderTime={MIN_SKELETONS_TIME}
                                    />

                                    {showSubscriptionAndDynamic &&
                                        (deviceType.isMobile ? (
                                            <AviaSubscriptionAndDynamicAsButtons
                                                dynamicsAreOpen={
                                                    dynamicsAreOpen
                                                }
                                                toggleDynamics={toggleDynamics}
                                                isSkeletons={
                                                    isInitialVariantsLoading
                                                }
                                                canMakeRequest={
                                                    !isInitialVariantsLoading
                                                }
                                                minLoaderTime={
                                                    MIN_SKELETONS_TIME
                                                }
                                                {...prepareQaAttributes({
                                                    parent: ROOT_QA,
                                                    current:
                                                        'subscriptionAndDynamicAsButtons',
                                                })}
                                            />
                                        ) : (
                                            <DesktopDynamics
                                                isSkeletons={
                                                    isInitialVariantsLoading
                                                }
                                                canMakeRequest={
                                                    !isInitialVariantsLoading
                                                }
                                                minLoaderTime={
                                                    MIN_SKELETONS_TIME
                                                }
                                                {...prepareQaAttributes({
                                                    parent: ROOT_QA,
                                                    current:
                                                        'subscriptionAndDynamicAsButtons',
                                                })}
                                            />
                                        ))}

                                    <AdBlock />

                                    {!lowerCrossSaleEnabled && crossSaleNode}

                                    <AviaSearchRestResults
                                        variants={searchVariantsView.rest}
                                        startsFrom={
                                            searchVariantsView.top.length
                                        }
                                        isLoading={isLoading}
                                        crossSaleNode={crossSaleNode}
                                        onUpdate={updateDirect}
                                    />

                                    {/* RestResults не покажет кросс-сейл, если там < 10 результатов или еще идет загрузка, поэтому рендерим тут */}
                                    {lowerCrossSaleEnabled &&
                                        searchVariantsView.rest.length > 0 &&
                                        (searchVariantsView.rest.length <
                                            SLICE_SIZE ||
                                            isLoading) &&
                                        crossSaleNode}

                                    <AviaResetFiltersBlock />

                                    <AviaSearchDisclaimers />
                                </Box>
                            </div>
                        </TwoColumnLayout.LeftColumn>
                        <TwoColumnLayout.RightColumn
                            className={cx('resultsRightColumn')}
                        >
                            {isWithoutDirectBlock ? (
                                <>
                                    {!aviaOnlyHotelsRight &&
                                        !aviaHotelsThenWeatherRight && (
                                            <AviaWeatherForecast
                                                items={weatherForecast.items}
                                                status={weatherForecast.status}
                                                className={cx(
                                                    'weatherForecast',
                                                )}
                                            />
                                        )}
                                    {!aviaOnlyWeatherRight && (
                                        <AviaHotelsCrossSaleMapSidebar />
                                    )}
                                    {!aviaOnlyHotelsRight &&
                                        aviaHotelsThenWeatherRight && (
                                            <AviaWeatherForecast
                                                items={weatherForecast.items}
                                                status={weatherForecast.status}
                                                className={cx(
                                                    'weatherForecast_bottom',
                                                )}
                                            />
                                        )}
                                </>
                            ) : (
                                <DirectWithUpdater
                                    className={cx(
                                        'resultsRightColumn__direct',
                                        {
                                            direct_slideUp: !isVisible,
                                        },
                                    )}
                                    ref={directRef}
                                />
                            )}
                        </TwoColumnLayout.RightColumn>
                    </TwoColumnLayout>
                </Container>
                <AviaSearchProgressBar
                    key={'progress_' + String(qid)}
                    visible={isLoading || isInitialVariantsLoading}
                    {...progress}
                />
                {!isInitialVariantsLoading && (
                    <UpdateNotification
                        key={'update_' + String(qid)}
                        forceHide={isBackground}
                    />
                )}
            </AviaSearchLayout>
        );
    },
);

interface IAviaSearchResultHookLayerProps {
    location: RouteComponentProps['location'];
    isBackground: boolean;
}

function AviaSearchResultHookLayer({
    location,
    isBackground,
}: IAviaSearchResultHookLayerProps): React.ReactElement {
    const {searchVariantsView, klass, variantsInfo, isRoundTrip} = useSelector(
        aviaSearchResultPageSelector,
    );

    const lowerCrossSaleEnabled = useSelector(lowerCrossSaleEnabledSelector);

    const {cont, qid, sorting, progress, error, searchKey, searchIsCompleted} =
        useSelector(aviaSearchResultState);

    const aviaContext = useSelector(aviaContextSelector);
    const fromCountryTitle = aviaContext.from?.countryTitle;
    const toCountryTitle = aviaContext.to?.countryTitle;

    const dispatch = useDispatch();
    const query = useMemo(() => {
        const search = parse(location.search);

        return {
            ...search,
            adult_seats: search.adult_seats || DEFAULT_ADULTS_COUNT,
            children_seats: search.children_seats || DEFAULT_CHILDREN_COUNT,
            infant_seats: search.infant_seats || DEFAULT_INFANTS_COUNT,
            when:
                typeof search.when === 'string'
                    ? parseAviaPeriod(search.when)
                    : search.when,
        };
    }, [location.search]) as ParsedQuery;
    const variantsAreEmpty = variantsInfo.all === 0;
    const noVisibleVariants =
        variantsAreEmpty || variantsInfo.all === variantsInfo.hidden;

    const rumUi = useRumUi();
    const initSearchCallback = useCallback(() => {
        dispatch(initSearch({query: query as any, rumUi}));
        dispatch(aviaOrderActions.setUseOrderData(false));
    }, [dispatch, query, rumUi]);

    useSearchMetrika(searchIsCompleted, searchVariantsView);

    useEffect(() => {
        dispatch(
            withPrevState(setAviaCurrentPage(EAviaActionLogPageName.SERP)),
        );
    }, [dispatch]);

    useEffect(() => {
        if (!isAviaSearchParams(query)) {
            return;
        }

        const nextKey = getAviaSearchParamsKey(query);

        if (
            !searchKey ||
            !isSameAviaSearchParamsKeys(searchKey, nextKey) ||
            !fromCountryTitle
        ) {
            initSearchCallback();
        }
    }, [fromCountryTitle, initSearchCallback, query, searchKey]);

    useRumObserveState(EAviaRumEvents.Sort, [searchVariantsView]);
    useRumObserveState(EAviaRumEvents.Filter, [searchVariantsView]);
    useRumObserveState(EAviaRumEvents.TDAnswerReceived, [searchVariantsView]);

    useEffect(() => {
        if (qid) {
            aviaLoggingBrowserProvider.userSearch(
                qid,
                EAviaUserSearchLogPages.SEARCH,
            );
            reachGoal(EAviaGoal.SEARCH_PAGE_SHOW);
        }
    }, [qid]);

    useEffect(() => {
        if (!noVisibleVariants) {
            reachGoal(EAviaGoal.SEARCH_SHOW_VARIANTS);
        }
    }, [qid, noVisibleVariants]);

    useEffect(() => Order.preload(), []);

    const directRef = useRef<IDirectWithUpdaterRef>(null);

    const updateDirect = React.useCallback(() => {
        if (!directRef.current) {
            return;
        }

        directRef.current.update();
    }, []);

    if (__SERVER__) {
        const redirectOptions = getAviaRedirectOptionsByQuery(query);

        if (redirectOptions) {
            return (
                <RedirectWithStatus
                    to={redirectOptions.path}
                    statusCode={redirectOptions.statusCode}
                />
            );
        }
    }

    return (
        <AviaSearchResultMemo
            cont={cont}
            qid={qid}
            searchVariantsView={searchVariantsView}
            progress={progress}
            searchIsCompleted={searchIsCompleted}
            error={error}
            klass={klass}
            directRef={directRef}
            noVisibleVariants={noVisibleVariants}
            isRoundTrip={isRoundTrip}
            updateDirect={updateDirect}
            initSearchCallback={initSearchCallback}
            isSortByInterest={sorting.type === EAviaResultsSortType.INTEREST}
            isBackground={isBackground}
            lowerCrossSaleEnabled={lowerCrossSaleEnabled}
            fromCountryTitle={fromCountryTitle}
            toCountryTitle={toCountryTitle}
        />
    );
}

export default serverFetchDataDispatcher([prefetchSearchData])(
    AviaSearchResultHookLayer,
);
