import React, {useCallback} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {resetFilterValues} from 'reducers/avia/search/results/filters/actions';

import {aviaContextSelector} from 'selectors/avia/context/aviaContextSelector';
import {aviaActiveFiltersCountSelector} from 'selectors/avia/search/filters/activeFilters';

import {isValidDate} from 'utilities/dateUtils';
import {deviceModMobile} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {EAviaRumEvents} from 'projects/avia/lib/EAviaRumEvents';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useExperiments} from 'utilities/hooks/useExperiments';

import Box from 'components/Box/Box';
import FiltersDesktop from 'components/FiltersDesktop/FiltersDesktop';
import {AviaSearchResultsFilters} from 'projects/avia/components/AviaSearchResultsFilters/AviaSearchResultsFilters';
import SearchResultsFiltersMobile from 'projects/avia/components/AviaSearchResultsFilters/components/SearchResultsFiltersMobile/SearchResultsFiltersMobile';

import {useRumUi} from 'contexts/RumUiContext';

import cx from './AviaSearchFilters.scss';

export interface IAviaSearchFiltersProps extends IWithQaAttributes {
    isSkeletons?: boolean;
    onChange?: () => void;
}

const SKELETONS_WIDTH = [180, 150, 106, 186, 108, 131, 100];

const ROOT_QA = 'avia-desktop-filters';

const AviaSearchFilters: React.FC<IAviaSearchFiltersProps> = ({
    isSkeletons,
    onChange,
    ...props
}) => {
    const context = useSelector(aviaContextSelector);
    const isRoundtrip = Boolean(
        context.return_date && isValidDate(context.return_date),
    );
    const deviceType = useDeviceType();
    const dispatch = useDispatch();
    const rumUi = useRumUi();

    const {aviaNewTouchFilters} = useExperiments();

    const activeFiltersCount = useSelector(aviaActiveFiltersCountSelector);

    const anyFilterIsActive = activeFiltersCount > 0;

    const handleReset = useCallback(() => {
        onChange?.();

        rumUi.measure(EAviaRumEvents.Filter);
        dispatch(resetFilterValues());
    }, [rumUi, dispatch, onChange]);

    if (deviceType.isMobile) {
        return (
            <Box
                x={4}
                className={cx('root', deviceModMobile('root', deviceType), {
                    root_withShadow: !aviaNewTouchFilters,
                })}
            >
                <SearchResultsFiltersMobile
                    isRoundtrip={isRoundtrip}
                    isSkeletons={isSkeletons}
                    onChange={onChange}
                    {...prepareQaAttributes(props)}
                />
            </Box>
        );
    }

    return (
        <FiltersDesktop
            className={cx('root', deviceModMobile('root', deviceType), {
                root_loaded: !isSkeletons,
            })}
            loading={isSkeletons}
            skeletonsWidth={SKELETONS_WIDTH}
            canReset={anyFilterIsActive}
            onReset={handleReset}
            resetButtonType="icon"
            withShadow={false}
            withoutHorizontalPaddings
            {...prepareQaAttributes(ROOT_QA)}
        >
            <AviaSearchResultsFilters
                isRoundtrip={isRoundtrip}
                onChange={onChange}
                {...prepareQaAttributes(ROOT_QA)}
            />
        </FiltersDesktop>
    );
};

export default AviaSearchFilters;
