import {useState, useEffect} from 'react';
import {parse} from 'query-string';
import {Location} from 'history';

import {isAviaSearchParams} from 'projects/avia/lib/search/isAviaSearchParams';
import dynamicsWeekDates from 'projects/avia/components/Dynamics/utilities/dynamicsWeekDates';
import isDateRobot from 'utilities/dateUtils/isDateRobot';

import {aviaPriceIndexBrowserProvider} from 'serviceProvider/avia/aviaPriceIndexBrowserProvider';
import aviaBrowserProvider from 'serviceProvider/avia/aviaBrowserProvider';

import {YakType} from '../YakType';

export function useYakType({search}: Location): YakType {
    const [yakType, setYakType] = useState(YakType.NONE);

    useEffect(() => {
        const params = parse(search);

        if (!isAviaSearchParams(params) || !isDateRobot(params.when)) {
            return;
        }

        let cancelled = false;

        const schedulePromise = aviaBrowserProvider.getSchedule(params);

        const days = dynamicsWeekDates(params.when);
        const {0: startDate, [days.length - 1]: endDate} = days;

        const pricesPromise = aviaPriceIndexBrowserProvider.dynamics(
            {startDate, endDate},
            params,
            {},
        );

        Promise.all([schedulePromise, pricesPromise]).then(
            ([schedule, prices]) => {
                if (cancelled) {
                    return;
                }

                if (schedule?.hasRoutes || Object.keys(prices).length) {
                    setYakType(YakType.DYNAMIC);
                } else {
                    setYakType(YakType.MAP);
                }
            },
            () => {
                if (cancelled) {
                    return;
                }

                setYakType(YakType.EMPTY);
            },
        );

        return () => {
            cancelled = true;
        };
    }, [search]);

    return yakType;
}
