import React, {useCallback, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {EAviaActionLogActionName} from 'server/loggers/avia/AviaActionLog/types/EAviaActionLogActionName';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {ESubscriptionSource} from 'types/subscription/ESubscriptionSource';

import {loggerActions} from 'reducers/avia/aviaLogging/actions';

import {IAviaVariantGroup} from 'selectors/avia/utils/denormalization/variantGroup';
import {lowerCrossSaleEnabledSelector} from 'selectors/avia/search/lowerCrossSaleEnabledSelector';
import {aviaCleanSerpEnabledSelector} from 'selectors/avia/search/aviaCleanSerpEnabledSelector';

import {usePrevious} from 'utilities/hooks/usePrevious';
import getAviaSliceKey from 'projects/avia/pages/AviaSearchResult/utilities/getAviaSliceKey';

import OptionalSubscription from 'containers/Subscriptions/Inline/OptionalSubscription';

import Box from 'components/Box/Box';
import {LazyList} from 'projects/avia/components/LazyList/LazyList';

import {useShowLogger} from 'projects/avia/pages/AviaSearchResult/useShowLogger';

import AviaVariantsSlice from '../AviaVariantsSlice/AviaVariantsSlice';

import cx from './AviaSearchRestResults.scss';

export const SLICE_SIZE = 10;

interface IAviaSearchRestResultsProps {
    variants: IAviaVariantGroup[];
    startsFrom: number;
    crossSaleNode: React.ReactNode;
    isLoading: boolean;
    onUpdate(): void;
}

const AviaSearchRestResults: React.FC<IAviaSearchRestResultsProps> = ({
    onUpdate,
    variants,
    isLoading,
    crossSaleNode,
    startsFrom,
}) => {
    const dispatch = useDispatch();
    const lowerCrossSaleEnabled = useSelector(lowerCrossSaleEnabledSelector);
    const aviaCleanSerpEnabled = useSelector(aviaCleanSerpEnabledSelector);
    const preparedVariants = useMemo(() => {
        if (isLoading) {
            return variants.slice(0, SLICE_SIZE);
        }

        return variants;
    }, [isLoading, variants]);

    const prevIsLoading = usePrevious(isLoading);

    const renderSliceCallback = useCallback(
        (slice: IAviaVariantGroup[]) => {
            return renderSlice(slice, prevIsLoading, isLoading);
        },
        [isLoading, prevIsLoading],
    );

    const renderSeparator = useCallback(
        (index: number) =>
            index === 0 &&
            !aviaCleanSerpEnabled && (
                <>
                    <OptionalSubscription
                        className={cx('subscription')}
                        vertical={ESubscriptionVerticalName.Avia}
                        source={ESubscriptionSource.SEARCH}
                        type="thin"
                    />

                    {lowerCrossSaleEnabled && crossSaleNode}
                </>
            ),
        [aviaCleanSerpEnabled, lowerCrossSaleEnabled, crossSaleNode],
    );

    const onRender = useShowLogger(startsFrom, false);
    const onScroll = useCallback(
        scrollPage => {
            dispatch(
                loggerActions.logSearchAction({
                    name: EAviaActionLogActionName.SEARCH_PAGE_SCROLL,
                    params: {
                        scrollPage,
                    },
                }),
            );
            onUpdate();
        },
        [dispatch, onUpdate],
    );

    return (
        <LazyList
            items={preparedVariants}
            renderSeparator={renderSeparator}
            renderSlice={renderSliceCallback}
            sliceSize={SLICE_SIZE}
            onRender={onRender}
            onScroll={onScroll}
        />
    );
};

function renderSlice(
    slice: IAviaVariantGroup[],
    prevIsLoading: boolean,
    isLoading: boolean,
): React.ReactNode {
    return (
        <Box
            below={3}
            key={getAviaSliceKey(
                prevIsLoading,
                slice[0] ? slice[0].key : 'empty',
            )}
        >
            <AviaVariantsSlice
                variants={slice}
                renderIsDynamic={!isLoading && !prevIsLoading}
            />
        </Box>
    );
}

export default AviaSearchRestResults;
