import React, {useCallback, useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import moment from 'moment';

import EAviaDynamicActionLogActionName from 'server/loggers/avia/AviaActionLog/types/EAviaDynamicActionLogActionName';
import EAsyncStatus from 'types/common/EAsyncStatus';

import weekPricesRequest from 'reducers/avia/aviaPriceIndex/weekPrices/weekPricesRequest';
import {loggerActions as aviaLoggingActions} from 'reducers/avia/aviaLogging/actions';

import getWeekPricesInfo from 'selectors/avia/weekPrices/aviaWeekPricesInfo';

import {EPriceHintType} from 'projects/avia/components/Dynamics/utilities/getPriceHintType';
import getDatesDelta from 'projects/avia/lib/dynamic/getDatesDelta';
import {CHAR_LIST_MARKER} from 'utilities/strings/charCodes';
import {HUMAN} from 'utilities/dateUtils/formats';
import isDateRobot from 'utilities/dateUtils/isDateRobot';
import getDynamicTitle from 'projects/avia/pages/AviaSearchResult/utilities/getDynamicTitle';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {
    dynamicDashPriceDashSubscribe,
    dynamicDashPriceDashSubscribed,
    dynamicDashShowDashAllDashCalendar,
    dynamicsDashDateDashDeparture,
    dynamicsDashDeltaDashDaysDashText,
    dynamicsDashShowDashAll,
} from 'i18n/avia-AviaDynamics';

import Card from 'components/Card/Card';
import Flex from 'components/Flex/Flex';
import AviaTrendM from 'icons/illustrations/AviaTrend/AviaTrendM';
import Text from 'components/Text/Text';
import MinLoaderTime from 'components/MinLoaderTime/MinLoaderTime';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import CalendarIcon from 'icons/16/Calendar';
import AviaSearchSubscribe, {
    IAviaSubscriptionRenderProps,
} from 'projects/avia/components/AviaSearchSubscribe';
import ArrowRightIcon from 'icons/16/ArrowRight';
import MailIcon from 'icons/16/Mail';
import Toggle from 'components/Toggle/Toggle';
import Skeleton from 'projects/avia/pages/AviaSearchResult/components/AviaSubscriptionAndDynamicAsButtons/Skeleton/Skeleton';
import BottomSheet from 'components/BottomSheet/BottomSheet';
import Days from 'projects/avia/pages/AviaSearchResult/components/AviaSubscriptionAndDynamicAsButtons/Days/Days';
import SearchDynamicSearchWrapper from 'projects/avia/components/Dynamics/SearchDynamic/SearchDynamicSearchWrapper';

import cx from './AviaSubscriptionAndDynamicAsButtons.scss';

interface IAviaSubscriptionAndDynamicAsButtonsProps extends IWithQaAttributes {
    dynamicsAreOpen: boolean;
    isSkeletons: boolean;
    minLoaderTime: number;
    toggleDynamics: () => void;
    canMakeRequest?: boolean;
}

function AviaSubscriptionAndDynamicAsButtons({
    dynamicsAreOpen,
    isSkeletons,
    minLoaderTime,
    toggleDynamics,
    canMakeRequest,
    ...rest
}: IAviaSubscriptionAndDynamicAsButtonsProps): React.ReactElement {
    const dispatch = useDispatch();
    const weekPriceInfo = useSelector(getWeekPricesInfo);
    const {
        requestParams,
        needRequest,
        actualStatus,
        priceType = EPriceHintType.EMPTY,
        priceDays,
        weekPrices,
    } = weekPriceInfo;
    const loading = isSkeletons || actualStatus !== EAsyncStatus.SUCCESS;
    const searchForm = requestParams?.searchForm;
    const when = searchForm?.when;
    const whenMoment = when && isDateRobot(when) ? moment(when) : moment();
    const dayDelta = searchForm ? getDatesDelta(searchForm) : null;
    const dateDeparture = whenMoment.format(HUMAN);

    const onOpenDynamics = useCallback(() => {
        toggleDynamics();
        dispatch(
            aviaLoggingActions.logDynamicAction({
                name: EAviaDynamicActionLogActionName.DYNAMIC_OPEN,
            }),
        );
    }, [dispatch, toggleDynamics]);

    const onCloseDynamics = useCallback(() => {
        toggleDynamics();
        dispatch(
            aviaLoggingActions.logDynamicAction({
                name: EAviaDynamicActionLogActionName.DYNAMIC_CLOSE,
            }),
        );
    }, [dispatch, toggleDynamics]);

    const subscribeElement = useCallback(function subscribeButton({
        subscribed,
        unsubscribe,
        toggleModal,
    }: IAviaSubscriptionRenderProps): React.ReactElement {
        return (
            <div
                className={cx('buttonContainer')}
                onClick={subscribed ? unsubscribe : toggleModal}
            >
                <div className={cx('buttonIconContainer')}>
                    <MailIcon />
                </div>

                <div className={cx('buttonRightContainer')}>
                    <Text size="s">
                        {subscribed
                            ? dynamicDashPriceDashSubscribed()
                            : dynamicDashPriceDashSubscribe()}
                    </Text>

                    <Toggle isActive={subscribed} />
                </div>
            </div>
        );
    },
    []);

    useEffect(() => {
        if (needRequest && requestParams && canMakeRequest) {
            dispatch(weekPricesRequest(requestParams));
        }
    }, [canMakeRequest, dispatch, needRequest, requestParams]);

    return (
        <Card
            x="3"
            y="5"
            textSize="m"
            shadow="default"
            below="3"
            className={cx('root')}
            {...prepareQaAttributes(rest)}
        >
            <MinLoaderTime
                isLoaderNeeded={loading}
                minLoaderTime={minLoaderTime}
                loaderNode={<Skeleton />}
                componentNode={
                    <>
                        <Flex below={4} alignItems="center">
                            <AviaTrendM width={32} height={32} />

                            <div className={cx('headerTextContainer')}>
                                <Text weight="medium" tag="div">
                                    {getDynamicTitle(priceType)}
                                </Text>

                                <div className={cx('daysText')}>
                                    {[
                                        dayDelta
                                            ? dynamicsDashDeltaDashDaysDashText(
                                                  {count: dayDelta},
                                              )
                                            : undefined,
                                        dynamicsDashDateDashDeparture({
                                            date: dateDeparture,
                                        }),
                                    ]
                                        .filter(Boolean)
                                        .join(`  ${CHAR_LIST_MARKER}  `)}
                                </div>
                            </div>
                        </Flex>

                        <HorizontalScroller offset={3}>
                            <div className={cx('daysContainer')}>
                                {requestParams && priceDays && weekPrices && (
                                    <Days
                                        requestParams={requestParams}
                                        priceDays={priceDays}
                                        weekPrices={weekPrices}
                                        onOpenDynamics={onOpenDynamics}
                                    />
                                )}

                                <Text
                                    size="s"
                                    className={cx('showAll')}
                                    onClick={onOpenDynamics}
                                >
                                    {dynamicsDashShowDashAll()}
                                </Text>
                            </div>
                        </HorizontalScroller>

                        <div
                            className={cx('buttonContainer')}
                            onClick={onOpenDynamics}
                        >
                            <div className={cx('buttonIconContainer')}>
                                <CalendarIcon />
                            </div>

                            <div className={cx('buttonRightContainer')}>
                                <Text size="s" color="link">
                                    {dynamicDashShowDashAllDashCalendar()}
                                </Text>

                                <ArrowRightIcon />
                            </div>
                        </div>

                        <div className={cx('separator')} />

                        <AviaSearchSubscribe
                            className={cx('button')}
                            renderTrigger={subscribeElement}
                        />
                    </>
                }
            />

            <BottomSheet isOpened={dynamicsAreOpen} onClose={onCloseDynamics}>
                <SearchDynamicSearchWrapper isVisible={dynamicsAreOpen} />
            </BottomSheet>
        </Card>
    );
}

export default React.memo(AviaSubscriptionAndDynamicAsButtons);
