import React, {useMemo} from 'react';

import TValidDataForRequestDynamic from 'types/avia/dynamic/TValidDataForRequestDynamic';

import {TDynamicsDaysInfo} from 'reducers/avia/aviaPriceIndex/utils/convertPriceIndexDataToDynamicsData';

import {getDynamicsCalendarDayType} from 'projects/avia/components/Dynamics/utilities/getDynamicsCalendarDayType';
import getNearestIndexes from 'projects/avia/lib/dynamic/getNearestIndexes';
import getFiltersForUrl from 'projects/avia/utilities/getFiltersForUrl';
import {IDayWithPrice} from 'projects/avia/components/Dynamics/utilities/getPriceHintInfo';
import getDatesDelta from 'projects/avia/lib/dynamic/getDatesDelta';

import DynamicDayButton from 'projects/avia/pages/AviaSearchResult/components/DynamicDayButton/DynamicDayButton';

import cx from './Days.scss';

export const MOBILE_DYNAMIC_NEAREST_DATES_TO_SHOW = 1;

export interface IDays {
    requestParams: TValidDataForRequestDynamic;
    priceDays: IDayWithPrice[];
    weekPrices: TDynamicsDaysInfo;
    onOpenDynamics: () => void;
}

const Days: React.FC<IDays> = ({
    requestParams,
    priceDays,
    weekPrices,
    onOpenDynamics,
}) => {
    const {searchForm, filters} = requestParams;
    const {when} = searchForm;
    const dayDelta = getDatesDelta(searchForm);

    const filtersForUrl = useMemo(() => getFiltersForUrl(filters), [filters]);

    const dynamicPriceDays = priceDays.map(day => ({
        ...day,
        ...weekPrices[day.date],
    }));
    const indexOfWhen = dynamicPriceDays.findIndex(({date}) => date === when);
    // Отображаем только две ближайшие даты к запрошенному дню
    const {start, end} = getNearestIndexes(
        dynamicPriceDays,
        indexOfWhen,
        MOBILE_DYNAMIC_NEAREST_DATES_TO_SHOW,
    );
    const dayPrices = dynamicPriceDays.map(({price}) => ({price}));

    const daysInfo = dynamicPriceDays
        .slice(start, end + 1)
        .map(({date, price, status}, index, slicedPriceDays) => {
            const datePriceType = getDynamicsCalendarDayType(
                {price: price ?? null},
                dayPrices,
            );
            const prevDate =
                index > 0 ? slicedPriceDays[index - 1].date : undefined;

            return {
                date,
                price,
                status,
                datePriceType,
                prevDate,
            };
        });

    return (
        <>
            {daysInfo.map(({date, datePriceType, price, status, prevDate}) => (
                <DynamicDayButton
                    key={date}
                    className={cx('day')}
                    datePriceType={datePriceType}
                    date={date}
                    dayDelta={dayDelta}
                    searchForm={searchForm}
                    filtersForUrl={filtersForUrl}
                    onOpenDynamics={onOpenDynamics}
                    price={price}
                    status={status}
                    prevDate={prevDate}
                />
            ))}
        </>
    );
};

export default React.memo(Days);
