import React, {useCallback, useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import moment from 'moment';

import EAviaDynamicActionLogActionName from 'server/loggers/avia/AviaActionLog/types/EAviaDynamicActionLogActionName';
import EAsyncStatus from 'types/common/EAsyncStatus';

import weekPricesRequest from 'reducers/avia/aviaPriceIndex/weekPrices/weekPricesRequest';
import {loggerActions as aviaLoggingActions} from 'reducers/avia/aviaLogging/actions';

import getWeekPricesInfo from 'selectors/avia/weekPrices/aviaWeekPricesInfo';

import getDatesDelta from 'projects/avia/lib/dynamic/getDatesDelta';
import {CHAR_LIST_MARKER} from 'utilities/strings/charCodes';
import {HUMAN} from 'utilities/dateUtils/formats';
import isDateRobot from 'utilities/dateUtils/isDateRobot';
import getDynamicTitle from 'projects/avia/pages/AviaSearchResult/utilities/getDynamicTitle';
import {EPriceHintType} from 'projects/avia/components/Dynamics/utilities/getPriceHintType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {
    dynamicDashPriceDashSubscribe,
    dynamicDashPriceDashSubscribed,
    dynamicDashShowDashAllDashCalendar,
    dynamicsDashDateDashDeparture,
    dynamicsDashDeltaDashDaysDashText,
    dynamicsDashUnsubscribe,
} from 'i18n/avia-AviaDynamics';

import Card from 'components/Card/Card';
import Flex from 'components/Flex/Flex';
import AviaTrendM from 'icons/illustrations/AviaTrend/AviaTrendM';
import Text from 'components/Text/Text';
import MinLoaderTime from 'components/MinLoaderTime/MinLoaderTime';
import CalendarIcon from 'icons/16/Calendar';
import AviaSearchSubscribe, {
    IAviaSubscriptionRenderProps,
} from 'projects/avia/components/AviaSearchSubscribe';
import MailIcon from 'icons/16/Mail';
import Skeleton from 'projects/avia/pages/AviaSearchResult/components/DesktopDynamics/Skeleton/Skeleton';
import Days from 'projects/avia/pages/AviaSearchResult/components/DesktopDynamics/Days';

import {useAviaUiContext} from 'projects/avia/context/AviaUiContext';

import cx from './DesktopDynamics.scss';

export interface IDesktopDynamics extends IWithQaAttributes {
    isSkeletons: boolean;
    minLoaderTime: number;
    canMakeRequest: boolean;
}

const DesktopDynamics: React.FC<IDesktopDynamics> = ({
    isSkeletons,
    minLoaderTime,
    canMakeRequest,
    ...rest
}) => {
    const dispatch = useDispatch();
    const {toggleDynamics, dynamicsIsVisible} = useAviaUiContext();
    const weekPriceInfo = useSelector(getWeekPricesInfo);
    const {
        requestParams,
        needRequest,
        actualStatus,
        priceType = EPriceHintType.EMPTY,
        priceDays,
        weekPrices,
    } = weekPriceInfo;
    const loading = isSkeletons || actualStatus !== EAsyncStatus.SUCCESS;
    const searchForm = requestParams?.searchForm;
    const when = searchForm?.when;
    const whenMoment = when && isDateRobot(when) ? moment(when) : moment();
    const dayDelta = searchForm ? getDatesDelta(searchForm) : null;
    const dateDeparture = whenMoment.format(HUMAN);

    const onOpenDynamics = useCallback(() => {
        if (dynamicsIsVisible) {
            return;
        }

        toggleDynamics();
        dispatch(
            aviaLoggingActions.logDynamicAction({
                name: EAviaDynamicActionLogActionName.DYNAMIC_OPEN,
            }),
        );
    }, [dispatch, dynamicsIsVisible, toggleDynamics]);

    const subscribeElement = useCallback(
        ({
            subscribed,
            unsubscribe,
            toggleModal,
        }: IAviaSubscriptionRenderProps): React.ReactElement => {
            return (
                <>
                    <div
                        className={cx('button', {
                            button_isSubscribed: subscribed,
                        })}
                        onClick={subscribed ? undefined : toggleModal}
                    >
                        <MailIcon className={cx('buttonIcon')} />

                        <span>
                            {subscribed
                                ? dynamicDashPriceDashSubscribed()
                                : dynamicDashPriceDashSubscribe()}
                        </span>
                    </div>

                    {subscribed && (
                        <div className={cx('button')} onClick={unsubscribe}>
                            <Text color="link">
                                {dynamicsDashUnsubscribe()}
                            </Text>
                        </div>
                    )}
                </>
            );
        },
        [],
    );

    useEffect(() => {
        if (needRequest && requestParams && canMakeRequest) {
            dispatch(weekPricesRequest(requestParams));
        }
    }, [canMakeRequest, dispatch, needRequest, requestParams]);

    return (
        <Card
            x="8"
            y="4"
            textSize="m"
            shadow="default"
            above="4"
            below="3"
            className={cx('root')}
            {...prepareQaAttributes(rest)}
        >
            <MinLoaderTime
                isLoaderNeeded={loading}
                minLoaderTime={minLoaderTime}
                loaderNode={<Skeleton />}
                componentNode={
                    <>
                        <Flex below={4} alignItems="center">
                            <AviaTrendM width={32} height={32} />

                            <div className={cx('headerTextContainer')}>
                                <Text weight="bold" tag="div" size="l">
                                    {getDynamicTitle(priceType)}
                                </Text>

                                <div className={cx('daysText')}>
                                    {[
                                        dayDelta
                                            ? dynamicsDashDeltaDashDaysDashText(
                                                  {count: dayDelta},
                                              )
                                            : undefined,
                                        dynamicsDashDateDashDeparture({
                                            date: dateDeparture,
                                        }),
                                    ]
                                        .filter(Boolean)
                                        .join(`  ${CHAR_LIST_MARKER}  `)}
                                </div>
                            </div>
                        </Flex>

                        {requestParams && priceDays && weekPrices && (
                            <Days
                                requestParams={requestParams}
                                priceDays={priceDays}
                                weekPrices={weekPrices}
                                onOpenDynamics={onOpenDynamics}
                            />
                        )}

                        <div className={cx('buttons')}>
                            <div
                                className={cx('button')}
                                onClick={onOpenDynamics}
                            >
                                <CalendarIcon className={cx('buttonIcon')} />

                                <Text color="link">
                                    {dynamicDashShowDashAllDashCalendar()}
                                </Text>
                            </div>

                            <AviaSearchSubscribe
                                renderTrigger={subscribeElement}
                            />
                        </div>
                    </>
                }
            />
        </Card>
    );
};

export default React.memo(DesktopDynamics);
