import React, {
    useEffect,
    useRef,
    useImperativeHandle,
    CSSProperties,
} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import {getAviaSearchFilters} from 'selectors/avia/search/aviaSearchResultsSelector';

import {useMobile} from 'utilities/hooks/useMobile';
import {directHelper} from 'projects/avia/lib/directHelper';
import {useTld} from 'utilities/hooks/useTld';
import getPlatform from 'utilities/deviceType/getPlatform';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import {Direct} from 'components/Direct/Direct';

interface IDirectWithUpdaterProps extends IWithClassName {
    style?: CSSProperties;
}

export interface IDirectWithUpdaterRef {
    update(): void;
}

const DirectWithUpdater = React.forwardRef<
    IDirectWithUpdaterRef,
    IDirectWithUpdaterProps
>(function DirectWithUpdater({className, style}, ref) {
    const filters = useSelector(getAviaSearchFilters);
    const platform = getPlatform(useMobile());
    const tld = useTld();
    const directRef = useRef<Direct | null>(null);

    const updateDirect = React.useCallback(() => {
        if (!directRef.current) {
            return;
        }

        directRef.current.update();
    }, []);

    useEffect(() => {
        window.addEventListener('focus', updateDirect);

        return (): void => {
            window.removeEventListener('focus', updateDirect);
        };
    }, [updateDirect]);

    useEffect(() => {
        updateDirect();
    }, [filters, updateDirect]);

    useImperativeHandle(ref, () => ({
        update: updateDirect,
    }));

    return (
        <Direct
            blockId={directHelper.getId(tld, platform, 'right')}
            className={className}
            ref={directRef}
            style={style}
            {...prepareQaAttributes('avia-search-direct')}
        />
    );
});

export default DirectWithUpdater;
