import React from 'react';
import moment from 'moment-timezone';

import TDateRobot from 'types/common/date/TDateRobot';
import {IAviaPriceIndexPrice} from 'server/api/AviaPriceIndexApi/types/IAviaPriceIndexPrice';
import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';
import {IUrlQuery} from 'utilities/url/types';
import {EDynamicsDayStatus} from 'projects/avia/components/Dynamics/types/IDynamicsDay';
import {IWithClassName} from 'types/withClassName';
import {EAviaLinkSource} from 'types/avia/url/EAviaLinkSource';

import {CHAR_MINUS} from 'utilities/strings/charCodes';
import {isHoliday} from 'utilities/dateUtils/isHoliday';
import getDateRobotFromMoment from 'utilities/dateUtils/getDateRobotFromMoment';
import {aviaURLs} from 'projects/avia/lib/urls';
import {EPriceHintType} from 'projects/avia/components/Dynamics/utilities/getPriceHintType';

import {
    dynamicsDashDayDashNoDashFlights,
    dynamicsDashUnknownDashPrice,
} from 'i18n/avia-AviaDynamics';

import FireIcon16 from 'icons/16/Fire';
import FireIcon24 from 'icons/24/Fire';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';

import cx from './DynamicDayButton.scss';

export interface IDynamicDayButton extends IWithClassName {
    datePriceType: EPriceHintType;
    date: TDateRobot;
    dayDelta: number | null;
    searchForm: IAviaParams;
    filtersForUrl: IUrlQuery;
    onOpenDynamics: () => void;
    price: IAviaPriceIndexPrice | null;
    status: EDynamicsDayStatus;
    prevDate: TDateRobot | undefined;

    size?: 'm' | 'l';
}

const DynamicDayButton: React.FC<IDynamicDayButton> = ({
    datePriceType,
    date,
    dayDelta,
    searchForm,
    filtersForUrl,
    onOpenDynamics,
    price,
    status,

    prevDate,
    size = 'm',
    className,
}) => {
    const startMoment = moment(date);
    const startDay = startMoment.format('D');
    const startWeekDay = startMoment.format('dd');
    const startWeekDayIsHoliday = Boolean(isHoliday(startMoment));
    const {when} = searchForm;

    const prevMoment = prevDate && moment(prevDate);
    const startMonth =
        prevMoment && prevMoment.format('M') !== startMoment.format('M')
            ? startMoment.format('MMM')
            : undefined;

    const returnMoment =
        dayDelta !== null
            ? startMoment.clone().add(dayDelta, 'day')
            : undefined;
    const returnDay = returnMoment?.format('D');
    const returnWeekDay = returnMoment?.format('dd');
    const returnWeekDayIsHoliday = Boolean(
        returnMoment && isHoliday(returnMoment),
    );
    const returnMonth =
        returnMoment &&
        returnMoment.format('M') !== startMoment.format('M') &&
        returnMoment.format('MMM');

    const isToday = date === when;
    const isMinimalPrice = datePriceType === EPriceHintType.MINIMAL;
    const isNoData = status === EDynamicsDayStatus.NO_DATA;
    const isSearchLink = !isToday && price && !price.roughly;

    const FireIcon = size === 'm' ? FireIcon16 : FireIcon24;

    const url = isSearchLink
        ? aviaURLs.getSearchResultsUrl(
              {
                  ...searchForm,
                  when: date,
                  return_date:
                      returnMoment && getDateRobotFromMoment(returnMoment),
                  linkSource: EAviaLinkSource.DYNAMIC,
              },
              filtersForUrl,
          )
        : undefined;

    const textWithoutPrice = isNoData
        ? dynamicsDashDayDashNoDashFlights()
        : dynamicsDashUnknownDashPrice();

    return (
        <a
            className={cx('day', className, {
                day_today: isToday,
                day_minimalPrice: isMinimalPrice,
                day_noData: isNoData,
                day_size_m: size === 'm',
                day_size_l: size === 'l',
            })}
            key={date}
            onClick={url ? undefined : onOpenDynamics}
            href={url}
            target="_blank"
            rel="noopener noreferrer"
        >
            <Text weight="bold" whiteSpace="nowrap" className={cx('price')}>
                {price ? (
                    <Price
                        {...price}
                        prefix={price.roughly ? '≈' : undefined}
                        prefixDelimiter={' '}
                    />
                ) : (
                    textWithoutPrice
                )}

                {isMinimalPrice && (
                    <>
                        {' '}
                        <FireIcon className={cx('fireIcon')} />
                    </>
                )}
            </Text>

            <Text tag="div" size="s" className={cx('dates')}>
                {startDay}{' '}
                {startWeekDayIsHoliday ? (
                    <span className={cx('holiday')}>{startWeekDay}</span>
                ) : (
                    startWeekDay
                )}{' '}
                {startMonth ? (
                    <span className={cx('month')}>{startMonth}</span>
                ) : (
                    false
                )}
                {returnDay ? ` ${CHAR_MINUS} ${returnDay}` : false}{' '}
                {returnWeekDayIsHoliday ? (
                    <span className={cx('holiday')}>{returnWeekDay}</span>
                ) : (
                    returnWeekDay
                )}{' '}
                {returnMonth ? (
                    <span className={cx('month')}>{returnMonth}</span>
                ) : (
                    false
                )}
            </Text>
        </a>
    );
};

export default React.memo(DynamicDayButton);
