import {useEffect} from 'react';
import {round} from 'lodash';

import {EFareFamilyAvailability} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';

import {IAviaSearchVariantsView} from 'selectors/avia/utils/getAviaSearchVariantsView';

import {usePrevious} from 'utilities/hooks/usePrevious';
import {params} from 'utilities/metrika';

export default function useSearchMetrika(
    searchIsCompleted: boolean,
    variants: IAviaSearchVariantsView,
): void {
    const prevSearchIsCompleted = usePrevious(searchIsCompleted);

    useEffect(() => {
        if (!prevSearchIsCompleted && searchIsCompleted) {
            const allSegments = [...variants.top, ...variants.rest];

            const segmentsWithFreeRefundCount = allSegments.filter(segment => {
                return segment.variants.some(
                    variant => variant.price.hasFreeRefund,
                );
            }).length;

            const segmentsWithFreeRefundRatio = round(
                segmentsWithFreeRefundCount / allSegments.length,
                2,
            );

            const segmentsWithChargedRefundCount = allSegments.filter(
                segment => {
                    return segment.variants.some(
                        variant => variant.price.hasChargedRefund,
                    );
                },
            ).length;

            const segmentsWithChargedRefundRatio = round(
                segmentsWithChargedRefundCount / allSegments.length,
                2,
            );

            const tariffWithChargedChangeRatioCount = allSegments.filter(
                segment => {
                    return segment.variants.some(
                        variant =>
                            variant.price.changingCarriage?.availability ===
                            EFareFamilyAvailability.charge,
                    );
                },
            ).length;

            const tariffWithChargedChangeRatio = round(
                tariffWithChargedChangeRatioCount / allSegments.length,
                2,
            );
            const tariffWithFreeChangeRatioCount = allSegments.filter(
                segment => {
                    return segment.variants.some(
                        variant =>
                            variant.price.changingCarriage?.availability ===
                            EFareFamilyAvailability.free,
                    );
                },
            ).length;

            const tariffWithFreeChangeRatio = Number(
                (tariffWithFreeChangeRatioCount / allSegments.length).toFixed(
                    2,
                ),
            );

            params({
                avia: {
                    tariffWithFreeRefundRatio: segmentsWithFreeRefundRatio,
                    tariffWithChargedRefundRatio:
                        segmentsWithChargedRefundRatio,
                    tariffWithChargedChangeRatio,
                    tariffWithFreeChangeRatio,
                },
            });
        }
    }, [prevSearchIsCompleted, searchIsCompleted, variants]);
}
