import React, {useEffect} from 'react';
import {useLocation, useParams} from 'react-router-dom';
import {parse} from 'query-string';
import {batch, useDispatch, useSelector} from 'react-redux';

import {ELoadableReducer} from 'types/common/ELoadableReducer';
import EAsyncStatus from 'types/common/EAsyncStatus';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';
import {IAviaSearchToAnywherePageParams} from 'projects/avia/pages/AviaSearchToAnywhere/types/IAviaSearchToAnywherePageParams';

import aviaSearchToAnywhereReducer from 'reducers/avia/searchToAnywhere/reducer';
import {requestSearchToAnywhereThunkAction} from 'reducers/avia/searchToAnywhere/requestSearchToAnywhereThunkAction';
import {resetAviaContext} from 'reducers/avia/context/actions';
import {reset} from 'reducers/avia/searchToAnywhere/actions';

import {
    searchToAnywhereRequestParamsSelector,
    searchToAnywhereSettlementsSelector,
    searchToAnywhereStatusSelector,
} from 'selectors/avia/searchToAnywhere/searchToAnywhereSelectors';

import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {isSettlementKey} from 'utilities/strings/isSettlementKey';
import {isStationKey} from 'utilities/strings/isStationKey';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {isCompatibleRequestParams} from 'projects/avia/lib/searchToAnywhere/isCompatibleRequestParams';
import {prefetchData} from 'projects/avia/pages/AviaSearchToAnywhere/utilities/prefetchData';
import {parseSearchParametersFromQuery} from 'projects/avia/lib/urls/parseSearchParametersFromQuery';

import withReducers from 'containers/withReducers/withReducers';

import AviaLayout from 'projects/avia/components/AviaLayout';
import NotFoundPage from 'components/NotFound/NotFoundPage/NotFoundPage';
import AviaSearchToAnywherePageContent from 'projects/avia/pages/AviaSearchToAnywhere/components/AviaSearchToAnywherePageContent/AviaSearchToAnywherePageContent';
import Error from 'projects/avia/pages/AviaLanding/components/Error/Error';
import AviaSearchHeaderFormDesktop from '../AviaSearchResult/components/AviaSearchHeaderForm/AviaSearchHeaderFormDesktop/AviaSearchHeaderFormDesktop';
import AviaSearchHeaderFormMobile from '../AviaSearchResult/components/AviaSearchHeaderForm/AviaSearchHeaderFormMobile/AviaSearchHeaderFormMobile';
import AviaSearchToAnywhereFilters from './components/AviaSearchToAnywhereFilters/AviaSearchToAnywhereFilters';
import Container from 'components/Layouts/Container/Container';

import {useAviaSearchToAnywhereFilters} from './hooks/useAviaSearchToAnywhereFilters';
import {useAviaSearchToAnywherePrepareSettlements} from './hooks/useAviaSearchToAnywherePrepareSettlements';
import {useAviaSearchToAnywhereHeader} from './hooks/useAviaSearchToAnywhereHeader';
import {useAviaSearchToAnywhereInlinePriceSearch} from './hooks/useAviaSearchToAnywhereInlinePriceSearch';
import {useAviaSearchToAnywherePrepareSettlementGroups} from './hooks/useAviaSearchToAnywherePrepareSettlementGroups';

const AviaSearchToAnywherePage: React.FC<IAviaSearchToAnywherePageParams> =
    props => {
        const dispatch = useDispatch();
        const needToRender = useServerDataFetcher([prefetchData(props)]);
        const {isMobile, isDesktop} = useDeviceType();
        const status = useSelector(searchToAnywhereStatusSelector);
        const requestParams = useSelector(
            searchToAnywhereRequestParamsSelector,
        );
        const settlements = useSelector(searchToAnywhereSettlementsSelector);

        const filters = useAviaSearchToAnywhereFilters();

        useEffect(() => {
            if (
                requestParams &&
                isCompatibleRequestParams(requestParams, props)
            ) {
                return;
            }

            dispatch(requestSearchToAnywhereThunkAction(props));
        }, [dispatch, props, requestParams]);

        useEffect(() => {
            return (): void => {
                batch(() => {
                    dispatch(reset());
                    // Чистим авиа-контекст, чтобы избежать некорректного поведения на других страницах
                    dispatch(resetAviaContext());
                });
            };
        }, [dispatch]);

        const AviaSearchHeaderForm = isMobile
            ? AviaSearchHeaderFormMobile
            : AviaSearchHeaderFormDesktop;

        const filteredSettlements = useAviaSearchToAnywherePrepareSettlements(
            filters.values,
            settlements,
        );

        const header = useAviaSearchToAnywhereHeader(filters.values);

        const settlementGroups = useAviaSearchToAnywherePrepareSettlementGroups(
            filteredSettlements,
            header,
            filters,
        );

        useAviaSearchToAnywhereInlinePriceSearch(settlementGroups);

        if (!needToRender) {
            return null;
        }

        if (status === EAsyncStatus.ERROR) {
            return <Error isNotFound={false} />;
        }

        return (
            <AviaLayout
                headerBorderBottomType={EHeaderBorderBottomType.NONE}
                hideSearchInformation={isDesktop}
            >
                <AviaSearchHeaderForm>
                    <AviaSearchToAnywhereFilters
                        {...filters}
                        isLoading={status === EAsyncStatus.LOADING}
                    />
                </AviaSearchHeaderForm>
                <Container>
                    <AviaSearchToAnywherePageContent
                        settlementGroups={settlementGroups}
                        // withoutTransfer={filters.values.withoutTransfers}
                    />
                </Container>
            </AviaLayout>
        );
    };

const AviaSearchToAnywherePageMemoized = React.memo(AviaSearchToAnywherePage);

// Слой с валидацией параметров
const AviaSearchToAnywherePageValidateParams: React.FC = () => {
    const {from} = useParams<{
        from: string;
    }>();

    const {search} = useLocation();
    const {dateForward, dateBackward, passengers, klass} =
        parseSearchParametersFromQuery(parse(search));

    if (!dateForward || (!isSettlementKey(from) && !isStationKey(from))) {
        return <NotFoundPage />;
    }

    return (
        <AviaSearchToAnywherePageMemoized
            from={from}
            dateForward={dateForward}
            dateBackward={dateBackward}
            passengers={passengers}
            klass={klass}
        />
    );
};

export default withReducers([
    [ELoadableReducer.AVIA_SEARCH_TO_ANYWHERE, aviaSearchToAnywhereReducer],
])(React.memo(AviaSearchToAnywherePageValidateParams));
