import {ReactElement, useCallback, useMemo, useRef, useState} from 'react';

import {
    ECountriesFilter,
    ECovid19Filter,
} from '../../types/IAviaSearchToAnywhereFilters';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/avia-SearchToAnywhere';

import DropdownFilter from 'components/DropdownFilter/DropdownFilter';
import FilterHeader from 'projects/avia/components/AviaSearchResultsFilters/components/FilterHeader/FilterHeader';
import Checkbox from 'components/Checkbox/Checkbox';
import Flex from 'components/Flex/Flex';
import CheckButton from 'components/CheckButton/CheckButton';
import BottomSheet from 'components/BottomSheet/BottomSheet';
import ButtonSkeleton from 'components/Skeletons/ButtonSkeleton/ButtonSkeleton';
import CloseIcon from 'icons/24/Close';
import Button from 'components/Button/Button';

import {IUseAviaSearchToAnywhereFilters} from '../../hooks/useAviaSearchToAnywhereFilters';

import cx from './AviaSearchToAnywhereFilters.scss';

export interface IAviaSearchToAnywhereFiltersParams
    extends IUseAviaSearchToAnywhereFilters {
    isAnyFilterApplied: boolean;
    isLoading?: boolean;
}

type TBottomSheetFiltersType = 'countries' | 'covid19' | null;

export default function AviaSearchToAnywhereFilters({
    values,
    isAnyFilterApplied,
    isLoading = false,
    onVisaFilterChange,
    onCountriesFilterReset,
    onCountriesFilterChange,
    onCovid19FilterReset,
    onCovid19FilterChange,
    onQuarantineFilterChange,
    onAcceptMirBankCardChange,
    onReset,
}: IAviaSearchToAnywhereFiltersParams): ReactElement {
    const deviceType = useDeviceType();

    const bottomSheetRef = useRef<HTMLDivElement>(null);

    const [showBottomSheetFilter, setShowBottomSheetFilter] =
        useState<TBottomSheetFiltersType>(null);
    const closeBottomSheetFilter = useCallback(
        () => setShowBottomSheetFilter(null),
        [],
    );
    const showCountriesBottomSheetFilter = useCallback(
        () => setShowBottomSheetFilter('countries'),
        [],
    );
    const showCovid19BottomSheetFilter = useCallback(
        () => setShowBottomSheetFilter('covid19'),
        [],
    );

    const spaceBetweenCheckboxes = deviceType.isDesktop ? 3 : 6;

    const countriesFilterContent = useMemo(
        () => (
            <>
                <FilterHeader
                    label={i18nBlock.countriesFilterTitle()}
                    buttonResetDisabled={values.countries.length === 0}
                    onReset={onCountriesFilterReset}
                />
                <Flex between={spaceBetweenCheckboxes} flexDirection="column">
                    <Checkbox
                        size="m"
                        label={i18nBlock.countriesFilterAll()}
                        checked={values.countries.includes(
                            ECountriesFilter.ALL,
                        )}
                        value={ECountriesFilter.ALL}
                        onChange={onCountriesFilterChange}
                        className={cx('countriesFilterAllCheckbox')}
                    />
                    <Checkbox
                        size="m"
                        label={i18nBlock.countriesFilterRussia()}
                        checked={values.countries.includes(
                            ECountriesFilter.RUSSIA,
                        )}
                        value={ECountriesFilter.RUSSIA}
                        onChange={onCountriesFilterChange}
                    />
                    <Checkbox
                        size="m"
                        label={i18nBlock.countriesFilterCis()}
                        checked={values.countries.includes(
                            ECountriesFilter.CIS,
                        )}
                        value={ECountriesFilter.CIS}
                        onChange={onCountriesFilterChange}
                    />
                    <Checkbox
                        size="m"
                        label={i18nBlock.countriesFilterEurope()}
                        checked={values.countries.includes(
                            ECountriesFilter.EUROPE,
                        )}
                        value={ECountriesFilter.EUROPE}
                        onChange={onCountriesFilterChange}
                    />
                    <Checkbox
                        size="m"
                        label={i18nBlock.countriesFilterAsia()}
                        checked={values.countries.includes(
                            ECountriesFilter.ASIA,
                        )}
                        value={ECountriesFilter.ASIA}
                        onChange={onCountriesFilterChange}
                    />
                    <Checkbox
                        size="m"
                        label={i18nBlock.countriesFilterAmerica()}
                        checked={values.countries.includes(
                            ECountriesFilter.AMERICA,
                        )}
                        value={ECountriesFilter.AMERICA}
                        onChange={onCountriesFilterChange}
                    />
                    <Checkbox
                        size="m"
                        label={i18nBlock.countriesFilterAfrica()}
                        checked={values.countries.includes(
                            ECountriesFilter.AFRICA,
                        )}
                        value={ECountriesFilter.AFRICA}
                        onChange={onCountriesFilterChange}
                    />
                </Flex>
            </>
        ),
        [
            onCountriesFilterChange,
            onCountriesFilterReset,
            spaceBetweenCheckboxes,
            values.countries,
        ],
    );

    const covid19FilterContent = useMemo(
        () => (
            <>
                <FilterHeader
                    label={i18nBlock.covid19FilterTitle()}
                    buttonResetDisabled={values.covid19.length === 0}
                    onReset={onCovid19FilterReset}
                />
                <Flex between={spaceBetweenCheckboxes} flexDirection="column">
                    <Checkbox
                        size="m"
                        label={i18nBlock.covid19FilterPcrTest()}
                        checked={values.covid19.includes(
                            ECovid19Filter.PCR_TEST,
                        )}
                        value={ECovid19Filter.PCR_TEST}
                        onChange={onCovid19FilterChange}
                    />
                    <Checkbox
                        size="m"
                        label={i18nBlock.covid19FilterSputnikV()}
                        checked={values.covid19.includes(
                            ECovid19Filter.SPUTNIK_V,
                        )}
                        value={ECovid19Filter.SPUTNIK_V}
                        onChange={onCovid19FilterChange}
                    />
                    <Checkbox
                        size="m"
                        label={i18nBlock.covid19FilterVaccineWho()}
                        checked={values.covid19.includes(
                            ECovid19Filter.VACCINE_WHO,
                        )}
                        value={ECovid19Filter.VACCINE_WHO}
                        onChange={onCovid19FilterChange}
                    />
                </Flex>
            </>
        ),
        [
            onCovid19FilterChange,
            onCovid19FilterReset,
            spaceBetweenCheckboxes,
            values.covid19,
        ],
    );

    const BottomSheetFilterContent = useMemo(() => {
        if (deviceType.isDesktop) return null;

        switch (showBottomSheetFilter) {
            case 'countries':
                return countriesFilterContent;
            case 'covid19':
                return covid19FilterContent;
            default:
                return null;
        }
    }, [
        deviceType.isDesktop,
        showBottomSheetFilter,
        countriesFilterContent,
        covid19FilterContent,
    ]);

    const BottomSheetFilter = useMemo(() => {
        if (deviceType.isDesktop) return null;

        return (
            <BottomSheet
                isOpened={showBottomSheetFilter !== null}
                onClose={closeBottomSheetFilter}
                paddingTopSpace={0}
                paddingBottomSpace={0}
                innerRef={bottomSheetRef}
                contentClassName={cx('bottomSheet')}
            >
                {BottomSheetFilterContent}
            </BottomSheet>
        );
    }, [
        BottomSheetFilterContent,
        closeBottomSheetFilter,
        deviceType,
        showBottomSheetFilter,
    ]);

    return (
        <div className={cx('filters', deviceMods('filters', deviceType))}>
            {isLoading ? (
                <>
                    {/*
                        * TODO: Без пересадок убираем пока бэк не начнет возвращать эти данные для всех городов
                        <ButtonSkeleton className={cx('skeleton')} withAnimation />
                    */}
                    <ButtonSkeleton className={cx('skeleton')} withAnimation />
                    <ButtonSkeleton className={cx('skeleton')} withAnimation />
                    <ButtonSkeleton className={cx('skeleton')} withAnimation />
                    <ButtonSkeleton className={cx('skeleton')} withAnimation />
                    <ButtonSkeleton className={cx('skeleton')} withAnimation />
                </>
            ) : (
                <>
                    {/*
                        * TODO: Без пересадок убираем пока бэк не начнет возвращать эти данные для всех городов
                        <CheckButton
                            size="m-inset"
                            checked={values.withoutTransfers}
                            onClick={onTransfersFilterChange}
                            width="max"
                            className={cx('filter')}
                        >
                            {i18nBlock.transfersFilterTitle()}
                        </CheckButton>
                    */}

                    <CheckButton
                        size="m-inset"
                        checked={values.withoutVisa}
                        onClick={onVisaFilterChange}
                        width="max"
                        className={cx('filter')}
                    >
                        {i18nBlock.visaFilterTitle()}
                    </CheckButton>

                    <DropdownFilter
                        label={i18nBlock.countriesFilterTitle()}
                        checked={Boolean(values.countries.length)}
                        popupClassName={cx('countryPopup')}
                        onClickButton={showCountriesBottomSheetFilter}
                        className={cx('filter')}
                    >
                        {deviceType.isDesktop && countriesFilterContent}
                    </DropdownFilter>

                    <DropdownFilter
                        label={i18nBlock.covid19FilterTitle()}
                        checked={Boolean(values.covid19.length)}
                        popupClassName={cx('covid19Popup')}
                        onClickButton={showCovid19BottomSheetFilter}
                        className={cx('filter')}
                    >
                        {deviceType.isDesktop && covid19FilterContent}
                    </DropdownFilter>

                    <CheckButton
                        size="m-inset"
                        checked={values.noQuarantine}
                        onClick={onQuarantineFilterChange}
                        width="max"
                        className={cx('filter')}
                    >
                        {i18nBlock.quarantineFilterTitle()}
                    </CheckButton>

                    <CheckButton
                        size="m-inset"
                        checked={values.acceptMirBankCard}
                        onClick={onAcceptMirBankCardChange}
                        width="max"
                        className={cx('filter')}
                    >
                        {i18nBlock.bankCardsFilterMir()}
                    </CheckButton>

                    {isAnyFilterApplied && (
                        <Button theme="attention" onClick={onReset}>
                            <CloseIcon />
                        </Button>
                    )}
                </>
            )}

            {BottomSheetFilter}
        </div>
    );
}
