import React, {useCallback, useEffect, useMemo, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {times} from 'lodash';

import EAsyncStatus from 'types/common/EAsyncStatus';
import {
    ECountryRestrictionsColor,
    IGetExtendedCountryRestrictionsResponse,
} from 'server/api/AviaCountryRestrictionsApi/types/TAviaCountryRestrictionsApiResponse';
import {EAviaLinkSource} from 'types/avia/url/EAviaLinkSource';
import {IAnywhereSettlementInfo} from 'types/avia/landing/ISettlementInfo';
import {ISettlementGroup} from '../../types/ISettlementGroup';

import {getCountryRestrictionsActions} from 'reducers/avia/countryRestrictions/actions';

import {
    searchToAnywhereFromPointSelector,
    searchToAnywhereRequestParamsSelector,
    searchToAnywhereStatusSelector,
} from 'selectors/avia/searchToAnywhere/searchToAnywhereSelectors';
import {getAviaCountryRestrictions} from 'selectors/avia/aviaSelectors';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {getSearchUrl} from 'projects/avia/lib/urls/getSearchUrl';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nSearchToAnywhere from 'i18n/avia-SearchToAnywhere';

import SettlementSkeleton from 'projects/avia/components/SettlementSkeleton/SettlementSkeleton';
import MinLoaderTime from 'components/MinLoaderTime/MinLoaderTime';
import CountryRestrictionsDetailed from 'components/CountryRestrictions/components/CountryRestrictionsDetailed/CountryRestrictionsDetailed';
import Link from 'components/Link/Link';
import Button from 'components/Button/Button';
import Flex from 'components/Flex/Flex';

import {SearchAnywhereSettlementGroup} from '../SearchAnywhereSettlementGroup/SearchAnywhereSettlementGroup';

import cx from './AviaSearchToAnywherePageContent.scss';

interface IAviaSearchToAnywherePageContentParams {
    settlementGroups: ISettlementGroup[];
    // TODO: Без пересадок убираем пока бэк не начнет возвращать эти данные для всех городов
    // withoutTransfer: boolean;
}

const AviaSearchToAnywherePageContent: React.FC<IAviaSearchToAnywherePageContentParams> =
    ({settlementGroups}) => {
        const dispatch = useDispatch();
        const deviceType = useDeviceType();

        const countryRestrictions = useSelector(getAviaCountryRestrictions);

        const fromPoint = useSelector(searchToAnywhereFromPointSelector);
        const requestParams = useSelector(
            searchToAnywhereRequestParamsSelector,
        );
        const status = useSelector(searchToAnywhereStatusSelector);

        const [selectedSettlement, setSelectedSettlement] =
            useState<IAnywhereSettlementInfo | null>(null);
        const [showSettlementResctrictions, setShowSettlementRescrictions] =
            useState(false);

        const onSettlementClick = useCallback(
            (settlement: IAnywhereSettlementInfo) => {
                setSelectedSettlement(settlement);
                setShowSettlementRescrictions(true);
            },
            [],
        );

        const closeSettlementRestrictionsModal = useCallback(() => {
            setShowSettlementRescrictions(false);
        }, []);

        useEffect(() => {
            if (!fromPoint?.key) return;

            if (
                selectedSettlement &&
                !countryRestrictions.cache?.[
                    selectedSettlement.geo_data.countryId
                ]
            ) {
                dispatch(
                    getCountryRestrictionsActions.request({
                        fromPointKey: fromPoint?.key,
                        toPointKey: selectedSettlement.key,
                        countryId: selectedSettlement.geo_data.countryId,
                    }),
                );
            } else {
                dispatch(getCountryRestrictionsActions.reset());
            }
        }, [
            selectedSettlement,
            fromPoint,
            dispatch,
            countryRestrictions.cache,
        ]);

        const content = useMemo(() => {
            return settlementGroups.map(({title, settlements, carousel}) => (
                <SearchAnywhereSettlementGroup
                    key={title}
                    header={title}
                    settlements={settlements}
                    isCarousel={carousel}
                    onSettlementClick={onSettlementClick}
                    headerClass={cx('header')}
                    containerClass={cx(
                        carousel
                            ? 'settlementContainerCarousel'
                            : 'settlementContainer',
                    )}
                />
            ));
        }, [settlementGroups, onSettlementClick]);

        const CountryRestrictionsDetailedFooter = useMemo(() => {
            if (!fromPoint || !requestParams) {
                return null;
            }

            if (!selectedSettlement) return null;

            const {dateForward, dateBackward, passengers, klass} =
                requestParams;

            const searchUrl = getSearchUrl({
                passengers,
                fromKey: fromPoint.key,
                toKey: selectedSettlement.key,
                klass,
                dateForward,
                dateBackward,
                // TODO: Без пересадок убираем пока бэк не начнет возвращать эти данные для всех городов
                // filters: withoutTransfer
                //                 ? {
                //                       transfer: {
                //                           value: createFlags(
                //                               EAviaSearchTransferFilter.NO_TRANSFERS,
                //                           ),
                //                           range: null,
                //                       },
                //                   }
                //                 : undefined,
                filters: undefined,
                linkSource: EAviaLinkSource.SEARCH_TO_ANYWHERE,
            });

            return (
                <Flex justifyContent="center">
                    <Link to={searchUrl} theme="black">
                        <Button
                            theme="primary"
                            className={cx('goToSearchButton')}
                            size="m"
                        >
                            {i18nSearchToAnywhere.goToSearchPage()}
                        </Button>
                    </Link>
                </Flex>
            );
        }, [fromPoint, requestParams, selectedSettlement]);

        const settlementRestrictions =
            useMemo((): IGetExtendedCountryRestrictionsResponse => {
                const countryId = selectedSettlement?.geo_data.countryId;

                if (countryId) {
                    const restrictions = countryRestrictions.cache?.[countryId];

                    if (restrictions) {
                        return {
                            ...restrictions,
                            titleText:
                                selectedSettlement.title.nominativeCase +
                                ' - ' +
                                restrictions.titleText,
                        };
                    }
                }

                const noInfoData: IGetExtendedCountryRestrictionsResponse = {
                    color: ECountryRestrictionsColor.GRAY,
                    titleText: '',
                    desktopMinimized: [],
                    mobileMinimized: [],
                    expandedBannerMetrics: [],
                };

                return noInfoData;
            }, [countryRestrictions.cache, selectedSettlement]);

        const skeleton = useMemo(
            () => (
                <div className={cx('settlementContainer')}>
                    {times(8, index => (
                        <SettlementSkeleton key={index} />
                    ))}
                </div>
            ),
            [],
        );

        return (
            <div className={cx('root', deviceMods('root', deviceType))}>
                <MinLoaderTime
                    minLoaderTime={2000}
                    isLoaderNeeded={status === EAsyncStatus.LOADING}
                    loaderNode={skeleton}
                    componentNode={content}
                />

                <CountryRestrictionsDetailed
                    countryRestrictions={settlementRestrictions}
                    isOpened={showSettlementResctrictions}
                    close={closeSettlementRestrictionsModal}
                    footer={CountryRestrictionsDetailedFooter}
                    loading={
                        countryRestrictions.status === EAsyncStatus.LOADING
                    }
                />
            </div>
        );
    };

export default React.memo(AviaSearchToAnywherePageContent);
