import {useCallback, useMemo, useState} from 'react';

import {
    getDefaultValues,
    ECountriesFilter,
    allCoutriesFilter,
    ECovid19Filter,
    IAviaSearchToAnywhereFiltersHanlders,
    IAviaSearchToAnywhereFilters,
} from '../types/IAviaSearchToAnywhereFilters';

export interface IUseAviaSearchToAnywhereFilters
    extends IAviaSearchToAnywhereFiltersHanlders {
    values: IAviaSearchToAnywhereFilters;
    isAnyFilterApplied: boolean;
}

export const useAviaSearchToAnywhereFilters =
    (): IUseAviaSearchToAnywhereFilters => {
        const [filters, setFilters] = useState(getDefaultValues());

        const scrollToTop = useCallback(() => {
            scrollTo({top: 0, behavior: 'smooth'});
        }, []);

        // TODO: Без пересадок убираем пока бэк не начнет возвращать эти данные для всех городов
        // const onTransfersFilterChange = useCallback(() => {
        //         setFilters({
        //             ...filters,
        //             withoutTransfers: !filters.withoutTransfers,
        //         });
        //         scrollToTop();
        //     }, [filters, scrollToTop]);

        const onVisaFilterChange = useCallback(() => {
            setFilters({
                ...filters,
                withoutVisa: !filters.withoutVisa,
            });
            scrollToTop();
        }, [filters, scrollToTop]);

        const onCountriesFilterReset = useCallback(() => {
            setFilters({
                ...filters,
                countries: [],
            });
            scrollToTop();
        }, [filters, scrollToTop]);

        const onCountriesFilterChange = useCallback(
            (e: React.ChangeEvent<HTMLInputElement>) => {
                const value = e.target.value as unknown as ECountriesFilter;

                let countries: ECountriesFilter[] = [];

                if (value === ECountriesFilter.ALL) {
                    countries = e.target.checked
                        ? [ECountriesFilter.ALL, ...allCoutriesFilter]
                        : filters.countries.filter(
                              country => country !== value,
                          );
                } else {
                    countries = e.target.checked
                        ? [...filters.countries, value]
                        : filters.countries.filter(
                              country =>
                                  country !== value &&
                                  country !== ECountriesFilter.ALL,
                          );
                }

                setFilters({
                    ...filters,
                    countries,
                });

                scrollToTop();
            },
            [filters, scrollToTop],
        );

        const onCovid19FilterReset = useCallback(() => {
            setFilters({...filters, covid19: []});
            scrollToTop();
        }, [filters, scrollToTop]);

        const onCovid19FilterChange = useCallback(
            (e: React.ChangeEvent<HTMLInputElement>) => {
                const value = Number(
                    e.target.value,
                ) as unknown as ECovid19Filter;

                const covid19 = e.target.checked
                    ? [...filters.covid19, value]
                    : filters.covid19.filter(
                          covidRestriction => covidRestriction !== value,
                      );

                setFilters({
                    ...filters,
                    covid19,
                });

                scrollToTop();
            },
            [filters, scrollToTop],
        );

        const onQuarantineFilterChange = useCallback(() => {
            setFilters({
                ...filters,
                noQuarantine: !filters.noQuarantine,
            });
            scrollToTop();
        }, [filters, scrollToTop]);

        const onAcceptMirBankCardChange = useCallback(() => {
            setFilters({
                ...filters,
                acceptMirBankCard: !filters.acceptMirBankCard,
            });
            scrollToTop();
        }, [filters, scrollToTop]);

        const onReset = useCallback(() => {
            setFilters(getDefaultValues());
            scrollToTop();
        }, [scrollToTop]);

        const isAnyFilterApplied = useMemo(
            () =>
                filters.withoutVisa ||
                Boolean(filters.countries.length) ||
                Boolean(filters.covid19.length) ||
                filters.noQuarantine ||
                filters.acceptMirBankCard,
            [filters],
        );

        return {
            values: filters,
            isAnyFilterApplied,
            onVisaFilterChange,
            onCountriesFilterReset,
            onCountriesFilterChange,
            onCovid19FilterReset,
            onCovid19FilterChange,
            onQuarantineFilterChange,
            onAcceptMirBankCardChange,
            onReset,
        };
    };
