import {useMemo} from 'react';

import {EFlightsAvailabilityV2} from 'server/api/AviaCountryRestrictionsApi/types/TAviaCountryRestrictionsApiResponse';
import {IAnywhereSettlementInfo} from 'types/avia/landing/ISettlementInfo';
import {ISettlementGroup} from '../types/ISettlementGroup';

import {russiaCountryId} from '../utilities/countryRegionMap';

import * as i18nBlock from 'i18n/avia-SearchToAnywhere';

import {IUseAviaSearchToAnywhereFilters} from './useAviaSearchToAnywhereFilters';

const MAX_SETTLEMENTS_GROUP_COUNT = 12;

export const useAviaSearchToAnywherePrepareSettlementGroups = (
    settlements: IAnywhereSettlementInfo[],
    header: string | null,
    filters: IUseAviaSearchToAnywhereFilters,
): ISettlementGroup[] => {
    return useMemo(() => {
        if (filters.isAnyFilterApplied) {
            const result: ISettlementGroup = {
                title: header,
                settlements: [],
                carousel: false,
            };

            // Для разнообразия выдачи решили:
            // для каждой выборки при применении фильтра или группы фильтров брать: 6 по России (если есть) и 6 заграничных (если есть)
            // Исключения - фильтр "банковские карты" или "без визы" - в этом случае стараемся показать все 12 городов НЕ российских (если есть)
            if (
                filters.values.acceptMirBankCard ||
                filters.values.withoutVisa
            ) {
                result.settlements = settlements
                    .sort((s1, s2) => {
                        const s1Weight =
                            s1.geo_data.countryId === russiaCountryId ? 0 : 1;
                        const s2Weight =
                            s2.geo_data.countryId === russiaCountryId ? 0 : 1;

                        return s2Weight - s1Weight;
                    })
                    .slice(0, MAX_SETTLEMENTS_GROUP_COUNT);
            } else {
                const {russianSettlements, foreignSettlements} =
                    settlements.reduce<{
                        russianSettlements: IAnywhereSettlementInfo[];
                        foreignSettlements: IAnywhereSettlementInfo[];
                    }>(
                        (res, settlement) => {
                            res[
                                settlement.geo_data.countryId ===
                                russiaCountryId
                                    ? 'russianSettlements'
                                    : 'foreignSettlements'
                            ].push(settlement);

                            return res;
                        },
                        {russianSettlements: [], foreignSettlements: []},
                    );

                const halfGroupCount = MAX_SETTLEMENTS_GROUP_COUNT / 2;

                if (russianSettlements.length < halfGroupCount) {
                    result.settlements = [
                        ...russianSettlements,
                        ...foreignSettlements.slice(
                            0,
                            MAX_SETTLEMENTS_GROUP_COUNT -
                                russianSettlements.length,
                        ),
                    ];
                } else if (foreignSettlements.length < halfGroupCount) {
                    result.settlements = [
                        ...russianSettlements.slice(
                            0,
                            MAX_SETTLEMENTS_GROUP_COUNT -
                                foreignSettlements.length,
                        ),
                        ...foreignSettlements,
                    ];
                } else {
                    result.settlements = [
                        ...russianSettlements.slice(0, halfGroupCount),
                        ...foreignSettlements.slice(0, halfGroupCount),
                    ];
                }
            }

            return [result];
        }

        // если не один фильтр не выбран выводим 3 группы (по россии, за границу без пересадок, без визы) по 12 городов в блоках с горизонтальным скролом
        const {
            russianSettlements,
            foreignCountriesWithoutTransfer,
            withoutVisa,
        } = settlements.reduce<{
            russianSettlements: IAnywhereSettlementInfo[];
            foreignCountriesWithoutTransfer: IAnywhereSettlementInfo[];
            withoutVisa: IAnywhereSettlementInfo[];
        }>(
            (result, settlement) => {
                if (settlement.geo_data.countryId === russiaCountryId) {
                    if (
                        result.russianSettlements.length <
                        MAX_SETTLEMENTS_GROUP_COUNT
                    )
                        result.russianSettlements.push(settlement);
                } else if (
                    settlement.country_restrictions.metrics
                        .flightsAvailabilityV2?.value ===
                    EFlightsAvailabilityV2.DIRECT_FLIGHTS
                ) {
                    if (
                        result.foreignCountriesWithoutTransfer.length <
                        MAX_SETTLEMENTS_GROUP_COUNT
                    )
                        result.foreignCountriesWithoutTransfer.push(settlement);
                } else if (
                    settlement.country_restrictions.metrics.visaRequired
                        ?.value === false
                ) {
                    if (result.withoutVisa.length < MAX_SETTLEMENTS_GROUP_COUNT)
                        result.withoutVisa.push(settlement);
                }

                return result;
            },
            {
                russianSettlements: [],
                foreignCountriesWithoutTransfer: [],
                withoutVisa: [],
            },
        );

        return [
            {
                title: i18nBlock.countriesFilterRussia(),
                settlements: russianSettlements,
                carousel: true,
            },
            {
                title: i18nBlock.foreignCountriesWithoutTransferFilter(),
                settlements: foreignCountriesWithoutTransfer,
                carousel: true,
            },
            {
                title: i18nBlock.visaFilterTitle(),
                settlements: withoutVisa,
                carousel: true,
            },
        ];
    }, [settlements, header, filters]);
};
