import React, {useEffect} from 'react';
import {useLocation, useParams} from 'react-router-dom';
import {parse} from 'query-string';
import {batch, useDispatch, useSelector} from 'react-redux';

import {ELoadableReducer} from 'types/common/ELoadableReducer';
import EAsyncStatus from 'types/common/EAsyncStatus';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';
import {IAviaSearchToCountryPageParams} from 'projects/avia/pages/AviaSearchToCountry/types/IAviaSearchToCountryPageParams';

import aviaSearchToCountryReducer from 'reducers/avia/searchToCountry/reducer';
import aviaCountryRestrictions from 'reducers/avia/countryRestrictions/reducer';
import {inlineSearchRequest} from 'reducers/avia/searchToCountry/inlineSearchRequest';
import requestSearchToCountryThunkAction from 'reducers/avia/searchToCountry/requestSearchToCountryThunkAction';
import {resetAviaContext} from 'reducers/avia/context/actions';
import {reset} from 'reducers/avia/searchToCountry/actions';

import {
    searchToCountryGetInlineSearchDataForCity,
    searchToCountryInlineSearchesSelector,
    searchToCountryRequestParamsSelector,
    searchToCountrySettlementsSelector,
    searchToCountryStatusSelector,
} from 'selectors/avia/searchToCountry/searchToCountrySelectors';

import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {isSettlementKey} from 'utilities/strings/isSettlementKey';
import {isStationKey} from 'utilities/strings/isStationKey';
import {isCountryKey} from 'utilities/strings/isCountryKey';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {isCompatibleRequestParams} from 'projects/avia/lib/searchToCountry/isCompatibleRequestParams';
import {prefetchData} from 'projects/avia/pages/AviaSearchToCountry/utilities/prefetchData';
import {parseSearchParametersFromQuery} from 'projects/avia/lib/urls/parseSearchParametersFromQuery';

import withReducers from 'containers/withReducers/withReducers';

import AviaLayout from 'projects/avia/components/AviaLayout';
import NotFoundPage from 'components/NotFound/NotFoundPage/NotFoundPage';
import AviaSearchToCountryPageContent from 'projects/avia/pages/AviaSearchToCountry/components/AviaSearchToCountryPageContent/AviaSearchToCountryPageContent';
import Error from 'projects/avia/pages/AviaLanding/components/Error/Error';
import Container from 'components/Layouts/Container/Container';

const AviaSearchToCountryPage: React.FC<IAviaSearchToCountryPageParams> =
    props => {
        const dispatch = useDispatch();
        const needToRender = useServerDataFetcher([prefetchData(props)]);
        const {isMobile} = useDeviceType();
        const status = useSelector(searchToCountryStatusSelector);
        const requestParams = useSelector(searchToCountryRequestParamsSelector);
        const settlements = useSelector(searchToCountrySettlementsSelector);
        const inlineSearches = useSelector(
            searchToCountryInlineSearchesSelector,
        );

        useEffect(() => {
            if (
                requestParams &&
                isCompatibleRequestParams(requestParams, props)
            ) {
                return;
            }

            dispatch(requestSearchToCountryThunkAction(props));
        }, [dispatch, props, requestParams]);

        // Запуск инлайн-поисков
        useEffect(() => {
            if (!requestParams) {
                return;
            }

            batch(() => {
                settlements.forEach(settlement => {
                    const inlineSearchData =
                        searchToCountryGetInlineSearchDataForCity(
                            inlineSearches,
                            settlement.key,
                        );
                    const priceFromPriceIndexIsExact = Boolean(
                        settlement.price && !settlement.price.expired,
                    );
                    const inlineSearchStarted = Boolean(inlineSearchData);

                    if (!priceFromPriceIndexIsExact && !inlineSearchStarted) {
                        dispatch(
                            inlineSearchRequest(requestParams, settlement.key),
                        );
                    }
                });
            });
        }, [requestParams, settlements, inlineSearches, dispatch]);

        useEffect(() => {
            return (): void => {
                batch(() => {
                    dispatch(reset());
                    // Чистим авиа-контекст, чтобы избежать некорректного поведения на других страницах
                    dispatch(resetAviaContext());
                });
            };
        }, [dispatch]);

        if (!needToRender) {
            return null;
        }

        if (status === EAsyncStatus.ERROR) {
            return <Error isNotFound={false} />;
        }

        return (
            <AviaLayout
                headerBorderBottomType={
                    isMobile ? EHeaderBorderBottomType.NONE : undefined
                }
            >
                <Container>
                    <AviaSearchToCountryPageContent />
                </Container>
            </AviaLayout>
        );
    };

const AviaSearchToCountryPageMemoized = React.memo(AviaSearchToCountryPage);

// Слой с валидацией параметров
const AviaSearchToCountryPageValidateParams: React.FC = () => {
    const {from, to} = useParams<{
        from: string;
        to: string;
    }>();

    const {search} = useLocation();
    const {dateForward, dateBackward, passengers, klass} =
        parseSearchParametersFromQuery(parse(search));

    if (
        !dateForward ||
        (!isSettlementKey(from) && !isStationKey(from)) ||
        !isCountryKey(to)
    ) {
        return <NotFoundPage />;
    }

    return (
        <AviaSearchToCountryPageMemoized
            from={from}
            to={to}
            dateForward={dateForward}
            dateBackward={dateBackward}
            passengers={passengers}
            klass={klass}
        />
    );
};

export default withReducers([
    [ELoadableReducer.AVIA_SEARCH_TO_COUNTRY, aviaSearchToCountryReducer],
    [ELoadableReducer.AVIA_COUNTRY_RESTRICTIONS, aviaCountryRestrictions],
])(React.memo(AviaSearchToCountryPageValidateParams));
