import React, {useMemo} from 'react';
import {useSelector} from 'react-redux';
import moment from 'moment';

import EAsyncStatus from 'types/common/EAsyncStatus';
import {EAviaLinkSource} from 'types/avia/url/EAviaLinkSource';

import {
    searchToCountryFromPointSelector,
    searchToCountryInlineSearchDataForCitySelector,
    searchToCountryRequestParamsSelector,
    searchToCountrySettlementsSelector,
    searchToCountryStatusSelector,
    searchToCountryToCountrySelector,
} from 'selectors/avia/searchToCountry/searchToCountrySelectors';
import {getAviaCountryRestrictions} from 'selectors/avia/aviaSelectors';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {CHAR_EM_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';
import {HUMAN} from 'utilities/dateUtils/formats';

import SettlementLink from 'projects/avia/components/Settlement/SettlementLink';
import SettlementSkeleton from 'projects/avia/components/SettlementSkeleton/SettlementSkeleton';
import CountryRestrictions from 'components/CountryRestrictions/CountryRestrictions';
import MinLoaderTime from 'components/MinLoaderTime/MinLoaderTime';
import Heading from 'components/Heading/Heading';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';
import CountryRestrictionsSkeleton from 'components/CountryRestrictions/components/CountryRestrictionsSkeleton/CountryRestrictionsSkeleton';

import cx from './AviaSearchToCountryPageContent.scss';

const SkeletonContent: React.FC = () => {
    const {isDesktop} = useDeviceType();

    return (
        <>
            {isDesktop && (
                <>
                    <TextSkeleton
                        className={cx('header_skeleton')}
                        size="xxl"
                    />

                    <TextSkeleton
                        className={cx('dates', 'dates_skeleton')}
                        size="l"
                    />
                </>
            )}

            <CountryRestrictionsSkeleton
                className={cx(
                    'countryRestrictions',
                    'countryRestrictions_skeleton',
                )}
            />

            <div className={cx('settlementContainer')}>
                {Array(8)
                    .fill(null)
                    .map((value, index) => (
                        <SettlementSkeleton key={index} />
                    ))}
            </div>
        </>
    );
};

const AviaSearchToCountryPageContent: React.FC = () => {
    const {isDesktop} = useDeviceType();
    const fromPoint = useSelector(searchToCountryFromPointSelector);
    const toCountry = useSelector(searchToCountryToCountrySelector);
    const requestParams = useSelector(searchToCountryRequestParamsSelector);
    const settlements = useSelector(searchToCountrySettlementsSelector);
    const status = useSelector(searchToCountryStatusSelector);
    const countryRestrictions = useSelector(getAviaCountryRestrictions);

    const content = useMemo(() => {
        if (!fromPoint || !toCountry || !requestParams) {
            return null;
        }

        const {dateForward, dateBackward, passengers, klass} = requestParams;

        const momentForward = moment(dateForward);
        const momentBackward = dateBackward && moment(dateBackward);
        /**
         * Не совсем верное время, так как правильнее всего было использовать
         * текущее время пункта отправления, но его нет, а большой точности нам здесь не нужно
         */
        const nowYear = moment().format('YYYY');
        const forwardYear = momentForward.format('YYYY');
        const backwardYear = momentBackward?.format('YYYY');
        const needShowYears = Boolean(
            nowYear !== forwardYear ||
                (backwardYear && forwardYear !== backwardYear),
        );

        return (
            <>
                {isDesktop && (
                    <>
                        <Heading level={1}>
                            {fromPoint.title.nominativeCase}
                            {CHAR_NBSP}
                            {CHAR_EM_DASH}
                            {CHAR_NBSP}
                            {toCountry.title.nominativeCase}
                        </Heading>

                        <div className={cx('dates')}>
                            {moment(dateForward).format(HUMAN)}
                            {needShowYears && `${CHAR_NBSP}${forwardYear}`}
                            {Boolean(dateBackward) && (
                                <>
                                    {CHAR_NBSP}
                                    {CHAR_EM_DASH}
                                    {CHAR_NBSP}
                                    {moment(dateBackward).format(HUMAN)}
                                    {needShowYears &&
                                        `${CHAR_NBSP}${backwardYear}`}
                                </>
                            )}
                        </div>
                    </>
                )}
                <CountryRestrictions
                    className={cx('countryRestrictions')}
                    countryRestrictions={countryRestrictions}
                    fromCountryTitle={fromPoint.countryTitle}
                    toCountryTitle={toCountry.title.nominativeCase}
                    borderRadius={isDesktop ? 'm' : 's'}
                />

                <div className={cx('settlementContainer')}>
                    {settlements.map(({key, title, images, price}) => (
                        <SettlementLink
                            key={key}
                            cityKey={key}
                            title={title.nominativeCase}
                            images={images}
                            minWidth={isDesktop ? 285 : 190}
                            fromPointKey={fromPoint.key}
                            klass={klass}
                            passengers={passengers}
                            dateForward={dateForward}
                            dateBackward={dateBackward}
                            size={isDesktop ? 'm' : 's'}
                            price={price}
                            inlineSearchDataForCitySelector={
                                searchToCountryInlineSearchDataForCitySelector
                            }
                            linkSource={EAviaLinkSource.SEARCH_TO_COUNTRY}
                        />
                    ))}
                </div>
            </>
        );
    }, [
        countryRestrictions,
        fromPoint,
        isDesktop,
        requestParams,
        settlements,
        toCountry,
    ]);

    return (
        <div className={cx('root', {root_isDesktop: isDesktop})}>
            <MinLoaderTime
                minLoaderTime={2000}
                isLoaderNeeded={status === EAsyncStatus.LOADING}
                loaderNode={<SkeletonContent />}
                componentNode={content}
            />
        </div>
    );
};

export default React.memo(AviaSearchToCountryPageContent);
